<?php
// upload_profile_image.php - (v3) محسّن مع حذف الصورة القديمة
// يستقبل صورة دائرية جاهزة من التطبيق ويحفظها مباشرة

ini_set('display_errors', 1);
error_reporting(E_ALL);
header('Content-Type: application/json; charset=utf-8');

require_once 'db_connect.php';

// ★★★ الإعدادات ★★★
define('UPLOAD_DIR', 'uploads/');
define('BASE_URL', 'https://latrotech.online/Copel/');
define('MAX_FILE_SIZE', 5242880); // 5 MB
define('ALLOWED_TYPES', ['jpg', 'jpeg', 'png', 'gif', 'webp']);

try {
    // ★★★ 1. التحقق من الاتصال بقاعدة البيانات ★★★
    if (!isset($conn)) {
        throw new Exception("فشل الاتصال بقاعدة البيانات.");
    }

    // ★★★ 2. التحقق من المدخلات ★★★
    $user_id = $_POST['user_id'] ?? null;
    
    if (empty($user_id) || !is_numeric($user_id)) {
        throw new Exception("معرّف المستخدم مطلوب ويجب أن يكون رقماً.");
    }
    
    $user_id = intval($user_id);

    // ★★★ 3. التحقق من وجود الملف المرفوع ★★★
    if (!isset($_FILES['profile_image']) || $_FILES['profile_image']['error'] != UPLOAD_ERR_OK) {
        $error_message = match($_FILES['profile_image']['error'] ?? -1) {
            UPLOAD_ERR_INI_SIZE, UPLOAD_ERR_FORM_SIZE => "حجم الصورة كبير جداً.",
            UPLOAD_ERR_PARTIAL => "تم رفع الصورة جزئياً فقط.",
            UPLOAD_ERR_NO_FILE => "لم يتم رفع أي صورة.",
            UPLOAD_ERR_NO_TMP_DIR => "مجلد مؤقت مفقود على الخادم.",
            UPLOAD_ERR_CANT_WRITE => "فشل كتابة الملف على القرص.",
            default => "حدث خطأ غير معروف أثناء رفع الصورة."
        };
        throw new Exception($error_message);
    }

    // ★★★ 4. التحقق من حجم الملف ★★★
    if ($_FILES['profile_image']['size'] > MAX_FILE_SIZE) {
        throw new Exception("حجم الصورة كبير جداً. الحد الأقصى 5 ميجابايت.");
    }

    // ★★★ 5. التحقق من نوع الملف بشكل آمن ★★★
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $_FILES['profile_image']['tmp_name']);
    finfo_close($finfo);

    $allowed_mimes = [
        'image/jpeg',
        'image/jpg', 
        'image/png',
        'image/gif',
        'image/webp'
    ];

    if (!in_array($mime_type, $allowed_mimes)) {
        throw new Exception("نوع الملف غير مسموح به. الرجاء رفع صورة (JPG, PNG, GIF, WEBP).");
    }

    // تحديد الامتداد من MIME Type
    $extension_map = [
        'image/jpeg' => 'jpg',
        'image/jpg' => 'jpg',
        'image/png' => 'png',
        'image/gif' => 'gif',
        'image/webp' => 'webp'
    ];
    $file_extension = $extension_map[$mime_type] ?? 'jpg';

    // ★★★ 6. التحقق من وجود المستخدم وجلب الصورة القديمة ★★★
    $stmt = $conn->prepare("SELECT profile_image_url FROM users WHERE id = :id");
    $stmt->execute([':id' => $user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        throw new Exception("المستخدم غير موجود.");
    }

    $old_image_url = $user['profile_image_url'];

    // ★★★ 7. إنشاء اسم فريد للملف الجديد ★★★
    $file_name = 'user_' . $user_id . '_' . time() . '.' . $file_extension;
    $target_file = UPLOAD_DIR . $file_name;

    // ★★★ 8. التأكد من وجود مجلد الرفع ★★★
    if (!is_dir(UPLOAD_DIR)) {
        if (!mkdir(UPLOAD_DIR, 0755, true)) {
            throw new Exception("فشل إنشاء مجلد الرفع.");
        }
    }

    // ★★★ 9. نقل الملف المرفوع (الصورة الدائرية الجاهزة من التطبيق) ★★★
    if (!move_uploaded_file($_FILES['profile_image']['tmp_name'], $target_file)) {
        throw new Exception("فشل حفظ الصورة على الخادم.");
    }

    error_log("✅ Image saved successfully: " . $target_file);

    // ★★★ 10. تحديث رابط الصورة في قاعدة البيانات ★★★
    $full_image_url = BASE_URL . $target_file;

    $updateStmt = $conn->prepare("UPDATE users SET profile_image_url = :url WHERE id = :id");
    $updateStmt->execute([
        ':url' => $full_image_url,
        ':id' => $user_id
    ]);

    error_log("✅ Database updated with new image URL: " . $full_image_url);

    // ★★★ 11. حذف الصورة القديمة من السيرفر (إن وجدت) ★★★
    if (!empty($old_image_url) && $old_image_url !== $full_image_url) {
        // استخراج المسار المحلي من الـ URL
        $old_file_path = str_replace(BASE_URL, '', $old_image_url);
        
        // التحقق من أن الملف موجود وليس صورة افتراضية
        if (file_exists($old_file_path) && 
            strpos($old_file_path, 'uploads/') === 0 &&
            strpos($old_file_path, 'default') === false &&
            strpos($old_file_path, 'placeholder') === false) {
            
            if (unlink($old_file_path)) {
                error_log("✅ Old image deleted: " . $old_file_path);
            } else {
                error_log("⚠️ Failed to delete old image: " . $old_file_path);
            }
        }
    }

    // ★★★ 12. إرسال رد ناجح ★★★
    http_response_code(200);
    echo json_encode([
        'status' => 'success',
        'message' => 'تم تحديث صورة الملف الشخصي بنجاح.',
        'profile_image_url' => $full_image_url,
        'uploaded_at' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
    // ★★★ معالجة الأخطاء ★★★
    http_response_code(400);
    error_log("❌ Upload error: " . $e->getMessage());
    
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

exit();
?>