<?php
/**
 * unlock_shop_item.php - النسخة المحسّنة
 * ======================================
 * فتح/شراء عنصر من المتجر مع التحقق الكامل
 * 
 * @version 2.0
 * @date 2026-02-05
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL);
ini_set('display_errors', 0);

// التحقق من وجود ملف الاتصال
if (!file_exists('db_connect.php')) {
    http_response_code(500);
    exit(json_encode([
        'status' => 'error', 
        'message' => 'خطأ في الاتصال بقاعدة البيانات'
    ]));
}

require_once 'db_connect.php';

// محاولة تحميل نظام النقاط
$pointsSystemAvailable = false;
$useNewPointsSystem = false;

if (file_exists('points_manager_v2.php')) {
    try {
        require_once 'points_manager_v2.php';
        $pointsSystemAvailable = true;
        $useNewPointsSystem = true;
    } catch (Exception $e) {
        error_log("Failed to load points_manager_v2.php: " . $e->getMessage());
    }
}

if (!$pointsSystemAvailable && file_exists('points_manager.php')) {
    try {
        require_once 'points_manager.php';
        $pointsSystemAvailable = true;
    } catch (Exception $e) {
        error_log("Failed to load points_manager.php: " . $e->getMessage());
    }
}

// استقبال البيانات
$input = json_decode(file_get_contents('php://input'), true);
$userId = $input['user_id'] ?? $_POST['user_id'] ?? 0;
$itemId = $input['item_id'] ?? $_POST['item_id'] ?? 0;

// التحقق من صحة البيانات
if ($userId <= 0 || $itemId <= 0) {
    http_response_code(400);
    exit(json_encode([
        'status' => 'error', 
        'message' => 'بيانات غير صالحة. يجب توفير user_id و item_id.'
    ]));
}

try {
    // ========================================
    // 1. بدء Transaction مع Isolation Level
    // ========================================
    $conn->exec("SET TRANSACTION ISOLATION LEVEL SERIALIZABLE");
    $conn->beginTransaction();
    
    // ========================================
    // 2. جلب بيانات المستخدم (مع قفل)
    // ========================================
    $stmt = $conn->prepare(
        "SELECT id, level, coins, xp, display_name 
         FROM users 
         WHERE id = :uid 
         FOR UPDATE"
    );
    $stmt->execute([':uid' => $userId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        throw new Exception('المستخدم غير موجود');
    }
    
    $userLevel = (int)$user['level'];
    $userCoins = (int)$user['coins'];
    
    // ========================================
    // 3. جلب بيانات العنصر
    // ========================================
    $stmt = $conn->prepare(
        "SELECT * 
         FROM shop_items 
         WHERE id = :iid 
         AND is_active = TRUE"
    );
    $stmt->execute([':iid' => $itemId]);
    $item = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$item) {
        throw new Exception('العنصر غير موجود أو غير متاح');
    }
    
    // ========================================
    // 4. التحقق من عدم فتحه مسبقاً
    // ========================================
    $stmt = $conn->prepare(
        "SELECT id 
         FROM user_unlocked_items 
         WHERE user_id = :uid 
         AND item_id = :iid"
    );
    $stmt->execute([':uid' => $userId, ':iid' => $itemId]);
    
    if ($stmt->fetch()) {
        throw new Exception('العنصر مفتوح بالفعل');
    }
    
    // ========================================
    // 5. حساب السعر الفعلي (مع الخصم)
    // ========================================
    $originalPrice = (int)$item['coins_required'];
    $finalPrice = $originalPrice;
    $isOnSale = false;
    
    if ($item['is_on_sale'] && $item['discount_percentage'] > 0) {
        $saleEndDate = $item['sale_end_date'];
        if ($saleEndDate === null || strtotime($saleEndDate) > time()) {
            $discountPercent = (int)$item['discount_percentage'];
            $finalPrice = floor($originalPrice * (100 - $discountPercent) / 100);
            $isOnSale = true;
        }
    }
    
    // ========================================
    // 6. التحقق من شروط الفتح
    // ========================================
    $coinsRequired = $finalPrice;
    $levelRequired = (int)$item['level_required'];
    $unlockType = $item['unlock_type'];
    
    $canUnlock = false;
    $errorMessage = '';
    
    switch ($unlockType) {
        case 'free':
            $canUnlock = true;
            break;
            
        case 'coins':
            if ($userCoins >= $coinsRequired) {
                $canUnlock = true;
            } else {
                $needed = $coinsRequired - $userCoins;
                $errorMessage = "تحتاج إلى $needed عملة إضافية (السعر: $coinsRequired، لديك: $userCoins)";
            }
            break;
            
        case 'level':
            if ($userLevel >= $levelRequired) {
                $canUnlock = true;
            } else {
                $needed = $levelRequired - $userLevel;
                $errorMessage = "تحتاج إلى $needed مستوى إضافي (المطلوب: $levelRequired، لديك: $userLevel)";
            }
            break;
            
        case 'both':
            $coinsOk = $userCoins >= $coinsRequired;
            $levelOk = $userLevel >= $levelRequired;
            
            if ($coinsOk && $levelOk) {
                $canUnlock = true;
            } else {
                $issues = [];
                if (!$coinsOk) {
                    $needed = $coinsRequired - $userCoins;
                    $issues[] = "$needed عملة";
                }
                if (!$levelOk) {
                    $needed = $levelRequired - $userLevel;
                    $issues[] = "$needed مستوى";
                }
                $errorMessage = "تحتاج إلى: " . implode(' و ', $issues);
            }
            break;
            
        default:
            throw new Exception('نوع فتح غير معروف');
    }
    
    if (!$canUnlock) {
        throw new Exception($errorMessage);
    }
    
    // ========================================
    // 7. خصم العملات (إن لزم الأمر)
    // ========================================
    $coinsSpent = 0;
    
    if ($coinsRequired > 0) {
        $stmt = $conn->prepare(
            "UPDATE users 
             SET coins = coins - :coins 
             WHERE id = :uid 
             AND coins >= :coins"
        );
        
        $stmt->execute([
            ':coins' => $coinsRequired, 
            ':uid' => $userId
        ]);
        
        if ($stmt->rowCount() === 0) {
            throw new Exception('فشل خصم العملات. قد تكون العملات غير كافية.');
        }
        
        $coinsSpent = $coinsRequired;
    }
    
    // ========================================
    // 8. فتح العنصر
    // ========================================
    $stmt = $conn->prepare(
        "INSERT INTO user_unlocked_items (user_id, item_id, unlocked_at) 
         VALUES (:uid, :iid, NOW())"
    );
    $stmt->execute([
        ':uid' => $userId, 
        ':iid' => $itemId
    ]);
    
    // ========================================
    // 9. تسجيل الشراء (إن كان بعملات)
    // ========================================
    if ($coinsSpent > 0) {
        $stmt = $conn->prepare(
            "INSERT INTO shop_purchases_log 
             (user_id, item_id, coins_spent, user_level_at_purchase, was_on_sale, original_price) 
             VALUES (:uid, :iid, :coins, :level, :on_sale, :original)"
        );
        
        $stmt->execute([
            ':uid' => $userId,
            ':iid' => $itemId,
            ':coins' => $coinsSpent,
            ':level' => $userLevel,
            ':on_sale' => $isOnSale ? 1 : 0,
            ':original' => $originalPrice
        ]);
    }
    
    // ========================================
    // 10. مسح الكاش
    // ========================================
    if (function_exists('apcu_delete')) {
        $cacheKey = "shop_items_user_{$userId}_v2";
        apcu_delete($cacheKey);
    }
    
    // ========================================
    // 11. Commit Transaction
    // ========================================
    $conn->commit();
    
    // ========================================
    // 12. بناء الاستجابة
    // ========================================
    $response = [
        'status' => 'success',
        'message' => 'تم فتح العنصر بنجاح! 🎉',
        'data' => [
            'item' => [
                'id' => (int)$item['id'],
                'name_ar' => $item['item_name_ar'],
                'type' => $item['item_type'],
                'rarity' => $item['rarity']
            ],
            'remaining_coins' => $userCoins - $coinsSpent,
            'coins_spent' => $coinsSpent,
            'was_on_sale' => $isOnSale,
            'savings' => $isOnSale ? ($originalPrice - $finalPrice) : 0
        ]
    ];
    
    // رسالة خاصة للعناصر المميزة
    if ($item['rarity'] === 'legendary') {
        $response['message'] = '🌟 مبروك! حصلت على عنصر أسطوري! 🌟';
    } elseif ($item['rarity'] === 'epic') {
        $response['message'] = '⭐ رائع! حصلت على عنصر ملحمي! ⭐';
    }
    
    echo json_encode($response);

} catch (Exception $e) {
    // Rollback في حالة الخطأ
    if ($conn->inTransaction()) {
        $conn->rollBack();
    }
    
    // تسجيل الخطأ
    error_log("Unlock Item Error: " . $e->getMessage() . " | User: $userId | Item: $itemId");
    
    // تحديد رمز الاستجابة
    $httpCode = 400;
    if (strpos($e->getMessage(), 'غير موجود') !== false) {
        $httpCode = 404;
    } elseif (strpos($e->getMessage(), 'قاعدة البيانات') !== false) {
        $httpCode = 500;
    }
    
    http_response_code($httpCode);
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage(),
        'error_code' => 'UNLOCK_FAILED'
    ]);
}
?>