<?php
/**
 * PointsSystem.php - نظام النقاط والمستويات المتكامل
 * يدير جميع عمليات منح النقاط، حساب المستويات، وتتبع التقدم
 */

class PointsSystem {
    private $conn;
    
    // تعريف قيم النقاط لكل إجراء
    private const ACTIONS = [
        'daily_login' => ['xp' => 50, 'love_points' => 10],
        'send_signal' => ['xp' => 20, 'love_points' => 5],
        'add_memory' => ['xp' => 30, 'love_points' => 8],
        'complete_game_round' => ['xp' => 40, 'love_points' => 12],
        'send_messages' => ['xp' => 5, 'love_points' => 2], // لكل 10 رسائل
    ];
    
    public function __construct($db_connection) {
        $this->conn = $db_connection;
    }
    
    /**
     * منح نقاط للمستخدم بناءً على نوع الإجراء
     * 
     * @param int $userId معرف المستخدم
     * @param int $coupleId معرف الزوجين
     * @param string $actionType نوع الإجراء
     * @param int $multiplier مضاعف النقاط (افتراضياً 1)
     * @return array نتيجة العملية
     */
    public function awardPoints($userId, $coupleId, $actionType, $multiplier = 1) {
        if (!isset(self::ACTIONS[$actionType])) {
            return ['success' => false, 'message' => 'Invalid action type'];
        }
        
        $action = self::ACTIONS[$actionType];
        $xpToAdd = $action['xp'] * $multiplier;
        $lovePointsToAdd = $action['love_points'] * $multiplier;
        
        try {
            // تحديث نقاط المستخدم
            $userUpdated = $this->updateUserXP($userId, $xpToAdd);
            $coupleUpdated = $this->updateCoupleLovePoints($coupleId, $lovePointsToAdd);
            
            // تسجيل الإجراء
            $this->logAction($userId, $coupleId, $actionType, $xpToAdd, $lovePointsToAdd);
            
            // التحقق من ارتقاء المستوى
            $levelUp = $this->checkLevelUp($userId);
            
            return [
                'success' => $userUpdated && $coupleUpdated,
                'xp_earned' => $xpToAdd,
                'love_points_earned' => $lovePointsToAdd,
                'level_up' => $levelUp
            ];
        } catch (Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * تحديث نقاط الخبرة للمستخدم
     */
    private function updateUserXP($userId, $xpToAdd) {
        $stmt = $this->conn->prepare("UPDATE users SET xp = xp + :xp WHERE id = :id");
        $stmt->execute([':xp' => $xpToAdd, ':id' => $userId]);
        return $stmt->rowCount() > 0;
    }
    
    /**
     * تحديث نقاط الحب للزوجين
     */
    private function updateCoupleLovePoints($coupleId, $pointsToAdd) {
        $stmt = $this->conn->prepare("UPDATE couples SET love_points = love_points + :points WHERE id = :id");
        $stmt->execute([':points' => $pointsToAdd, ':id' => $coupleId]);
        return $stmt->rowCount() > 0;
    }
    
    /**
     * التحقق من ارتقاء المستوى
     */
    private function checkLevelUp($userId) {
        // الحصول على البيانات الحالية
        $stmt = $this->conn->prepare("SELECT level, xp FROM users WHERE id = :id");
        $stmt->execute([':id' => $userId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$result) {
            return false;
        }
        
        $currentLevel = $result['level'];
        $currentXP = $result['xp'];
        
        // حساب XP المطلوب للمستوى التالي (المعادلة البسيطة: المستوى × 100)
        $nextLevel = $currentLevel + 1;
        $xpRequired = $nextLevel * 100;
        
        // التحقق من الارتقاء
        if ($currentXP >= $xpRequired) {
            $stmt = $this->conn->prepare("UPDATE users SET level = :level WHERE id = :id");
            $stmt->execute([':level' => $nextLevel, ':id' => $userId]);
            
            return [
                'leveled_up' => true,
                'new_level' => $nextLevel,
                'xp_for_next' => ($nextLevel + 1) * 100
            ];
        }
        
        return false;
    }
    
    /**
     * الحصول على XP المطلوب للمستوى التالي
     */
    public function getXPForNextLevel($currentLevel) {
        return ($currentLevel + 1) * 100;
    }
    
    /**
     * التحقق من تسجيل الدخول اليومي ومنح النقاط
     */
    public function checkDailyLogin($userId, $coupleId) {
        try {
            $stmt = $this->conn->prepare("SELECT last_daily_reward FROM users WHERE id = :id");
            $stmt->execute([':id' => $userId]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$result) {
                return ['success' => false, 'message' => 'User not found'];
            }
            
            $lastReward = $result['last_daily_reward'];
            $today = date('Y-m-d');
            
            // إذا لم يحصل على المكافأة اليوم بعد
            if ($lastReward !== $today) {
                // تحديث تاريخ آخر مكافأة
                $stmt = $this->conn->prepare("UPDATE users SET last_daily_reward = :date WHERE id = :id");
                $stmt->execute([':date' => $today, ':id' => $userId]);
                
                // منح النقاط
                $result = $this->awardPoints($userId, $coupleId, 'daily_login');
                $result['message'] = 'Daily reward claimed successfully!';
                return $result;
            }
            
            return ['success' => false, 'message' => 'Already claimed today'];
        } catch (Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * تسجيل الإجراء في قاعدة البيانات
     */
    private function logAction($userId, $coupleId, $actionType, $xpEarned, $lovePointsEarned) {
        try {
            $stmt = $this->conn->prepare(
                "INSERT INTO user_actions_log (user_id, couple_id, action_type, xp_earned, love_points_earned) 
                 VALUES (:uid, :cid, :type, :xp, :lp)"
            );
            $stmt->execute([
                ':uid' => $userId,
                ':cid' => $coupleId,
                ':type' => $actionType,
                ':xp' => $xpEarned,
                ':lp' => $lovePointsEarned
            ]);
        } catch (Exception $e) {
            // تسجيل الأخطاء لكن عدم إيقاف العملية
            error_log("Failed to log action: " . $e->getMessage());
        }
    }
    
    /**
     * الحصول على معلومات النقاط الكاملة للمستخدم
     */
    public function getUserPointsInfo($userId, $coupleId) {
        try {
            // معلومات المستخدم
            $stmt = $this->conn->prepare("SELECT level, xp, last_daily_reward FROM users WHERE id = :id");
            $stmt->execute([':id' => $userId]);
            $userInfo = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$userInfo) {
                return ['success' => false, 'message' => 'User not found'];
            }
            
            // نقاط الزوجين
            $lovePoints = 0;
            if ($coupleId > 0) {
                $stmt = $this->conn->prepare("SELECT love_points FROM couples WHERE id = :id");
                $stmt->execute([':id' => $coupleId]);
                $coupleInfo = $stmt->fetch(PDO::FETCH_ASSOC);
                $lovePoints = $coupleInfo ? $coupleInfo['love_points'] : 0;
            }
            
            // XP المطلوب للمستوى التالي
            $currentLevel = $userInfo['level'];
            $currentXp = $userInfo['xp'];
            $xpForNext = $this->getXPForNextLevel($currentLevel);
            
            // حساب التقدم
            $currentLevelXP = $currentLevel * 100;
            $xpProgress = $currentXp - $currentLevelXP;
            $xpNeeded = $xpForNext - $currentLevelXP;
            $progressPercentage = ($xpNeeded > 0) ? ($xpProgress / $xpNeeded) * 100 : 100;
            
            // التحقق من إمكانية الحصول على المكافأة اليومية
            $canClaimDaily = ($userInfo['last_daily_reward'] !== date('Y-m-d'));
            
            return [
                'success' => true,
                'level' => (int)$currentLevel,
                'current_xp' => (int)$currentXp,
                'xp_for_next_level' => (int)$xpForNext,
                'xp_progress' => max(0, (int)$xpProgress),
                'xp_needed' => (int)$xpNeeded,
                'progress_percentage' => max(0, min(100, round($progressPercentage, 2))),
                'love_points' => (int)$lovePoints,
                'can_claim_daily' => $canClaimDaily
            ];
        } catch (Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * إنفاق نقاط الحب (للمتجر لاحقاً)
     */
    public function spendLovePoints($coupleId, $amount) {
        try {
            // التحقق من الرصيد
            $stmt = $this->conn->prepare("SELECT love_points FROM couples WHERE id = :id");
            $stmt->execute([':id' => $coupleId]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$result) {
                return ['success' => false, 'message' => 'Couple not found'];
            }
            
            if ($result['love_points'] < $amount) {
                return ['success' => false, 'message' => 'Insufficient love points'];
            }
            
            // خصم النقاط
            $stmt = $this->conn->prepare("UPDATE couples SET love_points = love_points - :amount WHERE id = :id");
            $stmt->execute([':amount' => $amount, ':id' => $coupleId]);
            
            return [
                'success' => true,
                'remaining_points' => $result['love_points'] - $amount
            ];
        } catch (Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
}
?>