<?php
/**
 * PointsManager v2.0 - مع نظام الحدود اليومية
 * =====================================
 * المدير المركزي لنظام النقاط والعملات مع حدود يومية لكل نشاط
 * 
 * الميزات الجديدة:
 * - حدود يومية لكل نشاط (Daily Limits)
 * - تتبع عدد مرات كل نشاط يومياً
 * - إعادة تعيين تلقائية عند بداية يوم جديد
 * - منع الاستغلال والتلاعب
 * 
 * @version 2.0
 * @date 2026-02-04
 */

// ==========================================
// خريطة المكافآت مع الحدود اليومية
// ==========================================
const REWARDS_MAP_V2 = [
    // النشاط                => [عملات, نقاط خبرة, الحد اليومي للمكافآت]
    // ملاحظة: الحد اليومي يطبق على المكافآت فقط، النشاط نفسه غير محدود
    'daily_login'           => ['coins' => 20,  'xp' => 20,  'daily_limit' => 1],      // مكافأة 1 مرة/يوم
    'send_signal'           => ['coins' => 2,   'xp' => 5,   'daily_limit' => 5],      // مكافأة لـ 5 إشارات/يوم
    'add_memory'            => ['coins' => 50,  'xp' => 100, 'daily_limit' => 1],      // مكافأة لذكرى 1/يوم
    'truth_game_round'      => ['coins' => 15,  'xp' => 30,  'daily_limit' => 10],     // مكافأة لـ 10 جولات/يوم
    'send_message'          => ['coins' => 1,   'xp' => 2,   'daily_limit' => 10],     // مكافأة لـ 10 رسائل/يوم
    'spend_time_in_app'     => ['coins' => 5,   'xp' => 10,  'daily_limit' => 60],     // مكافأة لـ 60 دقيقة/يوم
];

// ==========================================
// إعدادات رفع المستوى
// ==========================================
const LEVEL_UP_XP_MULTIPLIER_V2 = 100;

/**
 * منح المكافأة للمستخدم مع التحقق من الحد اليومي
 * 
 * @param PDO $conn - الاتصال بقاعدة البيانات
 * @param int $userId - معرف المستخدم
 * @param string $actionType - نوع النشاط
 * @return array - النتيجة مع البيانات المحدثة
 */
function grantRewardV2($conn, $userId, $actionType) {
    try {
        // 1. التحقق من صحة نوع النشاط
        if (!isset(REWARDS_MAP_V2[$actionType])) {
            return [
                'status' => 'error',
                'message' => 'نوع النشاط غير معروف: ' . $actionType,
                'limit_reached' => false
            ];
        }

        $reward = REWARDS_MAP_V2[$actionType];
        $dailyLimit = $reward['daily_limit'];

        // 2. التحقق من الحد اليومي
        $limitCheck = checkDailyLimit($conn, $userId, $actionType, $dailyLimit);
        
        if (!$limitCheck['can_proceed']) {
            return [
                'status' => 'limit_reached',
                'message' => $limitCheck['message'],
                'limit_reached' => true,
                'count_today' => $limitCheck['count_today'],
                'daily_limit' => $dailyLimit,
                'coins_earned' => 0,
                'xp_earned' => 0,
                'level_up' => false,
                'new_level' => 0
            ];
        }

        // 3. إضافة العملات والخبرة
        $stmt_user = $conn->prepare(
            "UPDATE users 
             SET coins = coins + :coins, 
                 xp = xp + :xp 
             WHERE id = :uid"
        );
        
        $stmt_user->execute([
            ':coins' => $reward['coins'],
            ':xp'    => $reward['xp'],
            ':uid'   => $userId
        ]);

        // 4. تسجيل النشاط في جدول التتبع اليومي
        recordDailyAction($conn, $userId, $actionType, $reward['coins'], $reward['xp']);

        // 5. التحقق من إمكانية رفع المستوى
        $levelUpResult = checkLevelUpV2($conn, $userId);

        // 6. جلب العدد الحالي بعد الإضافة
        $currentCount = $limitCheck['count_today'] + 1;

        return [
            'status' => 'success',
            'message' => 'تم منح المكافأة بنجاح',
            'coins_earned' => $reward['coins'],
            'xp_earned' => $reward['xp'],
            'level_up' => $levelUpResult['level_up'],
            'new_level' => $levelUpResult['new_level'],
            'limit_reached' => false,
            'count_today' => $currentCount,
            'daily_limit' => $dailyLimit,
            'remaining_today' => max(0, $dailyLimit - $currentCount)
        ];

    } catch (Exception $e) {
        return [
            'status' => 'error',
            'message' => 'خطأ في منح المكافأة: ' . $e->getMessage(),
            'limit_reached' => false
        ];
    }
}

/**
 * التحقق من الحد اليومي للنشاط
 * 
 * @param PDO $conn - الاتصال بقاعدة البيانات
 * @param int $userId - معرف المستخدم
 * @param string $actionType - نوع النشاط
 * @param int $dailyLimit - الحد اليومي
 * @return array - نتيجة الفحص
 */
function checkDailyLimit($conn, $userId, $actionType, $dailyLimit) {
    try {
        $today = date('Y-m-d');

        // جلب عدد مرات النشاط اليوم
        $stmt = $conn->prepare(
            "SELECT COUNT(*) as count 
             FROM daily_actions_log 
             WHERE user_id = :uid 
             AND action_type = :action 
             AND DATE(action_date) = :today"
        );
        
        $stmt->execute([
            ':uid' => $userId,
            ':action' => $actionType,
            ':today' => $today
        ]);
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $countToday = (int)($result['count'] ?? 0);

        // التحقق من تجاوز الحد
        if ($countToday >= $dailyLimit) {
            return [
                'can_proceed' => false,
                'count_today' => $countToday,
                'message' => 'لقد وصلت إلى الحد اليومي لهذا النشاط (' . $dailyLimit . ' مرات/يوم).'
            ];
        }

        return [
            'can_proceed' => true,
            'count_today' => $countToday,
            'message' => 'يمكن المتابعة'
        ];

    } catch (Exception $e) {
        error_log("Error checking daily limit: " . $e->getMessage());
        // في حالة الخطأ، نسمح بالمتابعة لعدم تعطيل التطبيق
        return [
            'can_proceed' => true,
            'count_today' => 0,
            'message' => 'تم السماح بالمتابعة (خطأ في الفحص)'
        ];
    }
}

/**
 * تسجيل النشاط في جدول التتبع اليومي
 * 
 * @param PDO $conn - الاتصال بقاعدة البيانات
 * @param int $userId - معرف المستخدم
 * @param string $actionType - نوع النشاط
 * @param int $coinsEarned - العملات المكتسبة
 * @param int $xpEarned - الخبرة المكتسبة
 */
function recordDailyAction($conn, $userId, $actionType, $coinsEarned, $xpEarned) {
    try {
        $stmt = $conn->prepare(
            "INSERT INTO daily_actions_log 
             (user_id, action_type, coins_earned, xp_earned, action_date) 
             VALUES (:uid, :action, :coins, :xp, NOW())"
        );
        
        $stmt->execute([
            ':uid' => $userId,
            ':action' => $actionType,
            ':coins' => $coinsEarned,
            ':xp' => $xpEarned
        ]);

    } catch (Exception $e) {
        error_log("Error recording daily action: " . $e->getMessage());
    }
}

/**
 * التحقق من إمكانية رفع المستوى
 * 
 * @param PDO $conn - الاتصال بقاعدة البيانات
 * @param int $userId - معرف المستخدم
 * @return array - حالة رفع المستوى
 */
function checkLevelUpV2($conn, $userId) {
    try {
        // جلب المستوى الحالي ونقاط الخبرة
        $stmt = $conn->prepare("SELECT level, xp FROM users WHERE id = :uid");
        $stmt->execute([':uid' => $userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$user) {
            return [
                'level_up' => false,
                'new_level' => 0,
                'message' => 'المستخدم غير موجود'
            ];
        }

        $currentLevel = (int)$user['level'];
        $currentXp = (int)$user['xp'];
        $xpNeededForNextLevel = $currentLevel * LEVEL_UP_XP_MULTIPLIER_V2;
        
        $leveledUp = false;
        $newLevel = $currentLevel;

        // التحقق من إمكانية رفع المستوى (يمكن رفع أكثر من مستوى)
        while ($currentXp >= $xpNeededForNextLevel && $xpNeededForNextLevel > 0) {
            $newLevel++;
            $currentXp -= $xpNeededForNextLevel;
            $xpNeededForNextLevel = $newLevel * LEVEL_UP_XP_MULTIPLIER_V2;
            $leveledUp = true;
        }

        // إذا تم رفع المستوى، نحدث قاعدة البيانات
        if ($leveledUp) {
            $stmt_update = $conn->prepare(
                "UPDATE users 
                 SET level = :level, 
                     xp = :xp 
                 WHERE id = :uid"
            );
            
            $stmt_update->execute([
                ':level' => $newLevel,
                ':xp'    => $currentXp,
                ':uid'   => $userId
            ]);
        }

        return [
            'level_up' => $leveledUp,
            'new_level' => $newLevel,
            'remaining_xp' => $currentXp
        ];

    } catch (Exception $e) {
        return [
            'level_up' => false,
            'new_level' => 0,
            'message' => 'خطأ في فحص المستوى: ' . $e->getMessage()
        ];
    }
}

/**
 * الحصول على معلومات المكافأة لنشاط معين
 * 
 * @param string $actionType - نوع النشاط
 * @return array|null - معلومات المكافأة أو null
 */
function getRewardInfoV2($actionType) {
    return REWARDS_MAP_V2[$actionType] ?? null;
}

/**
 * الحصول على إحصائيات الحدود اليومية للمستخدم
 * 
 * @param PDO $conn - الاتصال بقاعدة البيانات
 * @param int $userId - معرف المستخدم
 * @return array - إحصائيات جميع الأنشطة اليوم
 */
function getDailyLimitsStats($conn, $userId) {
    try {
        $today = date('Y-m-d');
        $stats = [];

        foreach (REWARDS_MAP_V2 as $actionType => $reward) {
            // جلب عدد المرات اليوم
            $stmt = $conn->prepare(
                "SELECT COUNT(*) as count 
                 FROM daily_actions_log 
                 WHERE user_id = :uid 
                 AND action_type = :action 
                 AND DATE(action_date) = :today"
            );
            
            $stmt->execute([
                ':uid' => $userId,
                ':action' => $actionType,
                ':today' => $today
            ]);
            
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $countToday = (int)($result['count'] ?? 0);

            $stats[$actionType] = [
                'count_today' => $countToday,
                'daily_limit' => $reward['daily_limit'],
                'remaining' => max(0, $reward['daily_limit'] - $countToday),
                'limit_reached' => $countToday >= $reward['daily_limit'],
                'coins_per_action' => $reward['coins'],
                'xp_per_action' => $reward['xp']
            ];
        }

        return [
            'status' => 'success',
            'stats' => $stats
        ];

    } catch (Exception $e) {
        return [
            'status' => 'error',
            'message' => 'خطأ في جلب الإحصائيات: ' . $e->getMessage()
        ];
    }
}

/**
 * تنظيف السجلات القديمة (يفضل تشغيله عبر Cron Job)
 * 
 * @param PDO $conn - الاتصال بقاعدة البيانات
 * @param int $daysToKeep - عدد الأيام للاحتفاظ بالسجلات (افتراضي: 30)
 */
function cleanupOldLogs($conn, $daysToKeep = 30) {
    try {
        $stmt = $conn->prepare(
            "DELETE FROM daily_actions_log 
             WHERE action_date < DATE_SUB(NOW(), INTERVAL :days DAY)"
        );
        
        $stmt->execute([':days' => $daysToKeep]);
        
        return [
            'status' => 'success',
            'rows_deleted' => $stmt->rowCount()
        ];

    } catch (Exception $e) {
        return [
            'status' => 'error',
            'message' => 'خطأ في التنظيف: ' . $e->getMessage()
        ];
    }
}

/**
 * الحصول على جميع المكافآت المتاحة
 * 
 * @return array - قائمة بجميع المكافآت
 */
function getAllRewardsV2() {
    return REWARDS_MAP_V2;
}
?>