<?php
header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'db_connect.php';

$pusher_available = false;
if (file_exists(__DIR__ . '/PusherHelper.php')) {
    try {
        require_once __DIR__ . '/PusherHelper.php';
        $pusher_available = function_exists('send_pusher_trigger');
    } catch (Exception $e) {
        error_log("PusherHelper error: " . $e->getMessage());
    }
}

$json_input = file_get_contents('php://input');
$data       = json_decode($json_input, true) ?: [];
$response   = [];

try {
    if (!isset($data['user_id']) || !isset($data['room_id'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'Missing required parameters']);
        exit;
    }

    $user_id = intval($data['user_id']);
    $room_id = intval($data['room_id']);

    // التحقق من وجود الغرفة
    $stmt = $conn->prepare("SELECT id, max_members FROM chat_rooms WHERE id = ?");
    $stmt->execute([$room_id]);
    $room = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$room) {
        echo json_encode(['status' => 'error', 'message' => 'Room not found']);
        exit;
    }
    $max_members = $room['max_members'];

    // التحقق من الحظر
    $stmt = $conn->prepare("SELECT id FROM room_banned_users WHERE room_id = ? AND user_id = ?");
    $stmt->execute([$room_id, $user_id]);
    if ($stmt->fetch()) {
        echo json_encode(['status' => 'error', 'message' => 'You are banned from this room']);
        exit;
    }

    // جلب بيانات المستخدم
    $user_profile_stmt = $conn->prepare("SELECT display_name, level, profile_image_url, frame_identifier, name_color, font_identifier, is_verified FROM users WHERE id = ?");
    $user_profile_stmt->execute([$user_id]);
    $user_profile = $user_profile_stmt->fetch(PDO::FETCH_ASSOC);
    $userName = $user_profile['display_name'] ?? 'مستخدم';

    // وقت الدخول الحالي — يُستخدم لتحديث joined_at وإنشاء رسالة النظام
    $now = date('Y-m-d H:i:s');

    // هل هو عضو موجود مسبقاً؟
    $stmt = $conn->prepare("SELECT id, role, is_online FROM room_members WHERE room_id = ? AND user_id = ?");
    $stmt->execute([$room_id, $user_id]);
    $existing_member = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($existing_member) {
        // ✅ تحديث is_online=1 و joined_at=NOW() حتى لا يرى الرسائل القديمة
        $update_stmt = $conn->prepare("UPDATE room_members SET is_online = 1, joined_at = ? WHERE id = ?");
        $update_stmt->execute([$now, $existing_member['id']]);

        // إرسال رسالة انضمام فقط إذا كان غير متصل
        if ($existing_member['is_online'] == 0) {
            $systemMessage = "$userName انضم للغرفة";

            $stmt_msg = $conn->prepare("INSERT INTO room_messages (room_id, user_id, message_text, is_system, created_at) VALUES (?, ?, ?, 1, ?)");
            $stmt_msg->execute([$room_id, $user_id, $systemMessage, $now]);
            $messageId = $conn->lastInsertId();

            if ($pusher_available && $user_profile) {
                try {
                    send_pusher_trigger(
                        "c0a2fe3c47f0775d0be1", "85330156aec9dd641145", "2109257", "mt1",
                        'presence-room-' . $room_id, 'new-message',
                        [
                            'id'               => (int)$messageId,
                            'room_id'          => $room_id,
                            'user_id'          => $user_id,
                            'user_name'        => $userName,
                            'user_level'       => (int)$user_profile['level'],
                            'user_image_url'   => $user_profile['profile_image_url'],
                            'frame_identifier' => $user_profile['frame_identifier'],
                            'name_color'       => $user_profile['name_color'],
                            'font_identifier'  => $user_profile['font_identifier'],
                            'is_verified'      => (int)$user_profile['is_verified'],
                            'message_text'     => $systemMessage,
                            'is_system'        => true,
                            'created_at'       => $now,
                        ]
                    );
                } catch (Exception $e) {
                    error_log("Pusher failed in join_room (reconnect): " . $e->getMessage());
                }
            }
        }

        $response['status']  = 'success';
        $response['message'] = $existing_member['is_online'] == 1 ? 'Already connected' : 'Reconnected successfully';
        $response['role']    = $existing_member['role'];
        if ($user_profile) {
            unset($user_profile['profile_image_url']);
            $response['user_profile'] = $user_profile;
        }
        echo json_encode($response);
        exit;
    }

    // عضو جديد كلياً — التحقق من امتلاء الغرفة
    $stmt = $conn->prepare("SELECT COUNT(*) as total FROM room_members WHERE room_id = ? AND is_online = 1");
    $stmt->execute([$room_id]);
    if ($stmt->fetch(PDO::FETCH_ASSOC)['total'] >= $max_members) {
        echo json_encode(['status' => 'error', 'message' => 'Room is full']);
        exit;
    }

    // تحديد الدور
    $stmt_total = $conn->prepare("SELECT COUNT(*) as all_members FROM room_members WHERE room_id = ?");
    $stmt_total->execute([$room_id]);
    $role = ($stmt_total->fetch(PDO::FETCH_ASSOC)['all_members'] == 0) ? 'owner' : 'member';

    // ✅ إدراج العضو الجديد مع joined_at = NOW() صراحةً
    $stmt = $conn->prepare("INSERT INTO room_members (room_id, user_id, role, is_online, joined_at) VALUES (?, ?, ?, 1, ?)");

    if ($stmt->execute([$room_id, $user_id, $role, $now])) {
        $systemMessage = "$userName انضم للغرفة";

        $stmt_msg = $conn->prepare("INSERT INTO room_messages (room_id, user_id, message_text, is_system, created_at) VALUES (?, ?, ?, 1, ?)");
        $stmt_msg->execute([$room_id, $user_id, $systemMessage, $now]);
        $messageId = $conn->lastInsertId();

        if ($pusher_available && $user_profile) {
            try {
                send_pusher_trigger(
                    "c0a2fe3c47f0775d0be1", "85330156aec9dd641145", "2109257", "mt1",
                    'presence-room-' . $room_id, 'new-message',
                    [
                        'id'               => (int)$messageId,
                        'room_id'          => $room_id,
                        'user_id'          => $user_id,
                        'user_name'        => $userName,
                        'user_level'       => (int)$user_profile['level'],
                        'user_image_url'   => $user_profile['profile_image_url'],
                        'frame_identifier' => $user_profile['frame_identifier'],
                        'name_color'       => $user_profile['name_color'],
                        'font_identifier'  => $user_profile['font_identifier'],
                        'is_verified'      => (int)$user_profile['is_verified'],
                        'message_text'     => $systemMessage,
                        'is_system'        => true,
                        'created_at'       => $now,
                    ]
                );
            } catch (Exception $e) {
                error_log("Pusher failed in join_room: " . $e->getMessage());
            }
        }

        $response['status']  = 'success';
        $response['message'] = 'Joined successfully';
        $response['role']    = $role;
        if ($user_profile) {
            unset($user_profile['profile_image_url']);
            $response['user_profile'] = $user_profile;
        }
    } else {
        $response['status']  = 'error';
        $response['message'] = 'Failed to join room';
    }

} catch (PDOException $e) {
    http_response_code(500);
    $response['status']  = 'error';
    $response['message'] = 'Database error: ' . $e->getMessage();
} catch (Exception $e) {
    http_response_code(500);
    $response['status']  = 'error';
    $response['message'] = 'Server error: ' . $e->getMessage();
}

echo json_encode($response);
?>