<?php
/**
 * get_user_profile.php — معدّل لنظام VIP + Admin Badges
 * ========================================================
 * التعديلات الجديدة:
 *   - إضافة admin badges (جدول badges + user_badges)
 *   - كل شارة تحتوي: id, name, icon_url, color, animation_type, animation_config
 */

header('Content-Type: application/json; charset=utf-8');
require 'db_connect.php';

function getRankByLevel($level) {
    if ($level >= 50) {
        return ['rank_name' => 'أسطورة الحب',  'rank_badge' => 'https://latrotech.online/Copel/lvle/badge50.png'];
    } elseif ($level >= 40) {
        return ['rank_name' => 'ملك القلوب',   'rank_badge' => 'https://latrotech.online/Copel/lvle/badge40.png'];
    } elseif ($level >= 30) {
        return ['rank_name' => 'نجم الحب',     'rank_badge' => 'https://latrotech.online/Copel/lvle/badge30.png'];
    } elseif ($level >= 20) {
        return ['rank_name' => 'قلب فضي',      'rank_badge' => 'https://latrotech.online/Copel/lvle/bade10.png'];
    } elseif ($level >= 10) {
        return ['rank_name' => 'عاشق متألق',   'rank_badge' => 'https://latrotech.online/Copel/lvle/bade10.png'];
    } else {
        return ['rank_name' => 'مبتدئ',        'rank_badge' => 'https://latrotech.online/Copel/lvle/badg1.png'];
    }
}

$data       = json_decode(file_get_contents('php://input'), true);
$viewer_id  = $data['viewer_id']  ?? 0;
$profile_id = $data['profile_id'] ?? 0;

if (empty($viewer_id) || empty($profile_id)) {
    http_response_code(400);
    exit(json_encode(['status' => 'error', 'message' => 'Viewer ID and Profile ID are required.']));
}

try {
    // ══════════════════════════════════════════
    // جلب بيانات المستخدم + حقول VIP
    // ══════════════════════════════════════════
    $stmt_user = $conn->prepare("
        SELECT
            id, username, display_name, profile_image_url,
            bio, name_color, font_identifier, frame_identifier,
            is_verified, level, xp, coins, couple_id, created_at,
            age, marital_status,
            vip_tier, vip_expires_at, vip_gif_url, vip_badges
        FROM users
        WHERE id = :profile_id
    ");
    $stmt_user->execute([':profile_id' => $profile_id]);
    $user = $stmt_user->fetch(PDO::FETCH_ASSOC);

    if (!$user) throw new Exception("User not found.");

    // ══════════════════════════════════════════
    // حساب حالة VIP
    // ══════════════════════════════════════════
    $vipTier     = $user['vip_tier'] ?? 'none';
    $vipExpires  = $user['vip_expires_at'];
    $vipIsActive = ($vipTier !== 'none')
        && ($vipExpires === null || strtotime($vipExpires) > time());

    $vipFrameUrl = $vipIsActive
        ? "https://latrotech.online/Copel/vip/frames/frame_{$vipTier}.gif"
        : null;

    $vipGifUrl = ($vipIsActive && $vipTier === 'gold' && !empty($user['vip_gif_url']))
        ? $user['vip_gif_url']
        : null;

    // شارات VIP الخاصة بنظام VIP
    $vipBadgesData = [];
    if ($vipIsActive && !empty($user['vip_badges'])) {
        $badgeIds = json_decode($user['vip_badges'], true) ?? [];
        if (!empty($badgeIds)) {
            $placeholders = implode(',', array_fill(0, count($badgeIds), '?'));
            $stmtVipBadges = $conn->prepare(
                "SELECT id, name, icon_url FROM vip_badges WHERE id IN ($placeholders) ORDER BY sort_order"
            );
            $stmtVipBadges->execute($badgeIds);
            $vipBadgesData = $stmtVipBadges->fetchAll(PDO::FETCH_ASSOC);
        }
    }

    // ══════════════════════════════════════════
    // ★ جلب Admin Badges (النظام الجديد)
    //   animation_type: gold_stars | silver_bubbles | pink_hearts | fire | blue_galaxy | custom
    //   animation_config: JSON object (فقط لـ custom)
    // ══════════════════════════════════════════
    $adminBadgesData = [];
    $stmtAdminBadges = $conn->prepare("
        SELECT
            b.id,
            b.name,
            b.description,
            b.icon_url,
            b.color,
            b.animation_type,
            b.animation_config
        FROM user_badges ub
        JOIN badges b ON b.id = ub.badge_id AND b.is_active = 1
        WHERE ub.user_id = :uid
        ORDER BY b.sort_order ASC, b.id ASC
    ");
    $stmtAdminBadges->execute([':uid' => $profile_id]);
    $rawBadges = $stmtAdminBadges->fetchAll(PDO::FETCH_ASSOC);

    foreach ($rawBadges as $badge) {
        // animation_config مخزن كـ JSON string — نحوله لـ array/object
        $animConfig = null;
        if (!empty($badge['animation_config'])) {
            $animConfig = json_decode($badge['animation_config'], true);
        }

        $adminBadgesData[] = [
            'id'               => (int)$badge['id'],
            'name'             => $badge['name'],
            'description'      => $badge['description'],
            'icon_url'         => $badge['icon_url'],
            'color'            => $badge['color'],
            'animation_type'   => $badge['animation_type'] ?? 'gold_stars',
            'animation_config' => $animConfig, // null للنماذج الجاهزة، object للـ custom
        ];
    }

    // ══════════════════════════════════════════
    // بيانات الشريك
    // ══════════════════════════════════════════
    $relationship_status = ($viewer_id == $profile_id) ? 'self' : 'other';
    $partner_data        = null;

    if ($user['couple_id']) {
        $stmt_partner = $conn->prepare("
            SELECT u.id, u.display_name, u.profile_image_url,
                   u.name_color, u.font_identifier, u.frame_identifier, u.is_verified
            FROM users u
            JOIN couples c ON c.id = :couple_id
            WHERE (c.user1_id = u.id OR c.user2_id = u.id)
              AND u.id != :profile_id
            LIMIT 1
        ");
        $stmt_partner->execute([':couple_id' => $user['couple_id'], ':profile_id' => $profile_id]);
        $partner_data = $stmt_partner->fetch(PDO::FETCH_ASSOC);
    }

    $rank_info = getRankByLevel((int)($user['level'] ?? 1));

    // ══════════════════════════════════════════
    // بناء الـ Response
    // ══════════════════════════════════════════
    $response_data = [
        'id'                => (int)$user['id'],
        'username'          => $user['username'],
        'display_name'      => $user['display_name'],
        'profile_image_url' => $user['profile_image_url'],
        'bio'               => $user['bio'],
        'name_color'        => $user['name_color'],
        'font_identifier'   => $user['font_identifier'],
        'frame_identifier'  => $user['frame_identifier'],
        'is_verified'       => (bool)$user['is_verified'],
        'created_at'        => $user['created_at'],
        'age'               => $user['age'] !== null ? (int)$user['age'] : null,
        'marital_status'    => $user['marital_status'],
        'level'             => (int)($user['level'] ?? 1),
        'xp'                => (int)($user['xp']    ?? 0),
        'coins'             => (int)($user['coins'] ?? 0),
        'rank_name'         => $rank_info['rank_name'],
        'rank_badge'        => $rank_info['rank_badge'],
        'relationship_status' => $relationship_status,
        'is_own_profile'      => ($relationship_status === 'self'),

        // VIP Fields
        'vip_tier'        => $vipIsActive ? $vipTier : 'none',
        'vip_is_active'   => $vipIsActive,
        'vip_expires_at'  => $vipExpires,
        'vip_frame_url'   => $vipFrameUrl,
        'vip_gif_url'     => $vipGifUrl,
        'vip_badges_data' => $vipBadgesData,

        // ★ Admin Badges الجديدة
        'badges'          => $adminBadgesData,

        'partner_id'                => $partner_data ? (int)$partner_data['id']           : null,
        'partner_display_name'      => $partner_data ? $partner_data['display_name']      : null,
        'partner_profile_image_url' => $partner_data ? $partner_data['profile_image_url'] : null,
        'partner_name_color'        => $partner_data ? $partner_data['name_color']        : null,
        'partner_font_identifier'   => $partner_data ? $partner_data['font_identifier']   : null,
        'partner_frame_identifier'  => $partner_data ? $partner_data['frame_identifier']  : null,
        'partner_is_verified'       => $partner_data ? (bool)$partner_data['is_verified'] : false,
    ];

    echo json_encode(['status' => 'success', 'data' => $response_data]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}
?>