<?php
/**
 * get_shop_items.php - النسخة المحدثة النهائية v2.2
 * ====================================
 * جلب عناصر المتجر مع التحقق من الصلاحيات والكاش
 * + إضافة حقل is_applied
 * 
 * @version 2.2
 * @date 2026-02-14
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL);
ini_set('display_errors', 0);

// التحقق من وجود ملف الاتصال
if (!file_exists('db_connect.php')) {
    http_response_code(500);
    exit(json_encode([
        'success' => false,
        'status' => 'error', 
        'message' => 'خطأ في الاتصال بقاعدة البيانات'
    ]));
}

require_once 'db_connect.php';

// استقبال البيانات
$input = json_decode(file_get_contents('php://input'), true);
$userId = $input['user_id'] ?? $_POST['user_id'] ?? 0;

// التحقق من صحة البيانات
if ($userId <= 0) {
    http_response_code(400);
    exit(json_encode([
        'success' => false,
        'status' => 'error', 
        'message' => 'معرف المستخدم مطلوب'
    ]));
}

try {
    // ========================================
    // 1. التحقق من وجود المستخدم وجلب بياناته
    // ========================================
    $stmt = $conn->prepare(
        "SELECT id, level, coins, xp, display_name 
         FROM users 
         WHERE id = :uid"
    );
    $stmt->execute([':uid' => $userId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        http_response_code(404);
        exit(json_encode([
            'success' => false,
            'status' => 'error', 
            'message' => 'المستخدم غير موجود'
        ]));
    }
    
    $userLevel = (int)$user['level'];
    $userCoins = (int)$user['coins'];
    
    // ========================================
    // 2. محاولة جلب البيانات من الكاش
    // ========================================
    $cacheKey = "shop_items_user_{$userId}_v3"; // v3 بسبب is_applied
    $cacheTime = 300; // 5 دقائق
    $useCache = false;
    
    if (function_exists('apcu_fetch')) {
        $cached = apcu_fetch($cacheKey, $success);
        if ($success) {
            // التحقق من أن بيانات المستخدم لم تتغير
            if ($cached['user_level'] == $userLevel && 
                $cached['user_coins'] == $userCoins) {
                echo json_encode($cached);
                exit;
            }
        }
        $useCache = true;
    }
    
    // ========================================
    // 3. جلب جميع عناصر المتجر النشطة
    // ========================================
    $stmt = $conn->prepare("
        SELECT 
            si.id,
            si.item_type,
            si.item_identifier,
            si.item_name,
            si.item_name_ar,
            si.item_description,
            si.item_preview_url,
            si.unlock_type,
            si.coins_required,
            si.level_required,
            si.is_premium,
            si.category,
            si.rarity,
            si.is_on_sale,
            si.discount_percentage,
            si.sale_end_date,
            
            -- التحقق من فتح العنصر
            CASE 
                WHEN uui.id IS NOT NULL THEN TRUE 
                ELSE FALSE 
            END as is_unlocked,
            
            -- ★★★ NEW: التحقق من التطبيق الحالي ★★★
            CASE 
                WHEN uai.id IS NOT NULL THEN TRUE 
                ELSE FALSE 
            END as is_applied,
            
            -- حساب السعر الفعلي (مع الخصم إن وجد)
            CASE 
                WHEN si.is_on_sale = TRUE 
                    AND si.discount_percentage > 0 
                    AND (si.sale_end_date IS NULL OR si.sale_end_date > NOW())
                THEN FLOOR(si.coins_required * (100 - si.discount_percentage) / 100)
                ELSE si.coins_required
            END as final_price,
            
            -- التحقق من إمكانية الفتح
            CASE 
                -- إذا كان مفتوحاً بالفعل
                WHEN uui.id IS NOT NULL THEN FALSE
                
                -- إذا كان مجانياً
                WHEN si.unlock_type = 'free' THEN TRUE
                
                -- إذا كان يتطلب عملات فقط
                WHEN si.unlock_type = 'coins' THEN
                    CASE 
                        WHEN si.is_on_sale = TRUE 
                            AND si.discount_percentage > 0 
                            AND (si.sale_end_date IS NULL OR si.sale_end_date > NOW())
                        THEN :coins >= FLOOR(si.coins_required * (100 - si.discount_percentage) / 100)
                        ELSE :coins >= si.coins_required
                    END
                
                -- إذا كان يتطلب مستوى فقط
                WHEN si.unlock_type = 'level' THEN :level >= si.level_required
                
                -- إذا كان يتطلب كليهما
                WHEN si.unlock_type = 'both' THEN
                    CASE 
                        WHEN si.is_on_sale = TRUE 
                            AND si.discount_percentage > 0 
                            AND (si.sale_end_date IS NULL OR si.sale_end_date > NOW())
                        THEN (:coins >= FLOOR(si.coins_required * (100 - si.discount_percentage) / 100) 
                              AND :level >= si.level_required)
                        ELSE (:coins >= si.coins_required AND :level >= si.level_required)
                    END
                
                ELSE FALSE
            END as can_unlock,
            
            -- التحقق من المفضلة
            CASE 
                WHEN ufi.id IS NOT NULL THEN TRUE 
                ELSE FALSE 
            END as is_favorite
            
        FROM shop_items si
        LEFT JOIN user_unlocked_items uui 
            ON si.id = uui.item_id AND uui.user_id = :uid
        LEFT JOIN user_applied_items uai 
            ON si.id = uai.item_id AND uai.user_id = :uid
        LEFT JOIN user_favorite_items ufi 
            ON si.id = ufi.item_id AND ufi.user_id = :uid
        WHERE si.is_active = TRUE
        ORDER BY 
            si.item_type,
            CASE si.rarity
                WHEN 'legendary' THEN 1
                WHEN 'epic' THEN 2
                WHEN 'rare' THEN 3
                WHEN 'common' THEN 4
                ELSE 5
            END,
            si.sort_order,
            si.id
    ");
    
    $stmt->execute([
        ':uid' => $userId,
        ':coins' => $userCoins,
        ':level' => $userLevel
    ]);
    
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // ========================================
    // 4. تنظيم البيانات حسب النوع
    // ========================================
    $organized = [
        'frames' => [],
        'fonts' => [],
        'colors' => [],
        'badges' => []
    ];
    
    $stats = [
        'total_items' => count($items),
        'unlocked_items' => 0,
        'applied_items' => 0,
        'can_unlock_items' => 0,
        'favorite_items' => 0,
        'on_sale_items' => 0
    ];
    
    foreach ($items as $item) {
        // تحويل القيم البوليانية
        $item['is_unlocked'] = (bool)$item['is_unlocked'];
        $item['is_applied'] = (bool)$item['is_applied'];
        $item['can_unlock'] = (bool)$item['can_unlock'];
        $item['is_favorite'] = (bool)$item['is_favorite'];
        $item['is_premium'] = (bool)$item['is_premium'];
        $item['is_on_sale'] = (bool)$item['is_on_sale'];
        
        // تحويل الأرقام
        $item['id'] = (int)$item['id'];
        $item['coins_required'] = (int)$item['coins_required'];
        $item['level_required'] = (int)$item['level_required'];
        $item['final_price'] = (int)$item['final_price'];
        $item['discount_percentage'] = (int)$item['discount_percentage'];
        
        // إضافة حالة العنصر (status)
        if ($item['is_unlocked']) {
            $item['status'] = 'unlocked';
        } elseif (!$item['can_unlock']) {
            // تحديد سبب القفل
            if ($userCoins < $item['coins_required']) {
                $item['status'] = 'insufficient_coins';
            } elseif ($userLevel < $item['level_required']) {
                $item['status'] = 'insufficient_level';
            } else {
                $item['status'] = 'locked';
            }
        } else {
            $item['status'] = 'available';
        }
        
        // إضافة معلومات الخصم إن وجد
        if ($item['is_on_sale'] && $item['discount_percentage'] > 0) {
            $item['original_price'] = $item['coins_required'];
            $item['savings'] = $item['coins_required'] - $item['final_price'];
        }
        
        // حذف حقول غير ضرورية للعميل
        unset($item['sale_end_date']);
        
        // إحصائيات
        if ($item['is_unlocked']) $stats['unlocked_items']++;
        if ($item['is_applied']) $stats['applied_items']++;
        if ($item['can_unlock']) $stats['can_unlock_items']++;
        if ($item['is_favorite']) $stats['favorite_items']++;
        if ($item['is_on_sale']) $stats['on_sale_items']++;
        
        // تنظيم حسب النوع
        $type = $item['item_type'] . 's';
        $organized[$type][] = $item;
    }
    
    // ========================================
    // 5. جلب التخصيصات الحالية للمستخدم
    // ========================================
    $stmt = $conn->prepare(
        "SELECT frame_identifier, font_identifier, name_color 
         FROM users 
         WHERE id = :uid"
    );
    $stmt->execute([':uid' => $userId]);
    $currentCustomizations = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // ========================================
    // 6. بناء الاستجابة
    // ========================================
    $response = [
        'success' => true,
        'status' => 'success',
        'user_level' => $userLevel,
        'user_coins' => $userCoins,
        'user_xp' => (int)$user['xp'],
        'items' => $organized,
        'stats' => $stats,
        'current_customizations' => [
            'frame' => $currentCustomizations['frame_identifier'] ?? null,
            'font' => $currentCustomizations['font_identifier'] ?? null,
            'color' => $currentCustomizations['name_color'] ?? null
        ],
        'cache_enabled' => $useCache
    ];
    
    // ========================================
    // 7. حفظ في الكاش
    // ========================================
    if ($useCache) {
        apcu_store($cacheKey, $response, $cacheTime);
    }
    
    // ========================================
    // 8. إرسال الاستجابة
    // ========================================
    echo json_encode($response);

} catch (PDOException $e) {
    // تسجيل الخطأ
    error_log("Shop Items Error (DB): " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'status' => 'error',
        'message' => 'خطأ في قاعدة البيانات',
        'error_code' => 'DB_ERROR'
    ]);
    
} catch (Exception $e) {
    // تسجيل الخطأ
    error_log("Shop Items Error: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'status' => 'error',
        'message' => 'خطأ في السيرفر',
        'error_code' => 'SERVER_ERROR'
    ]);
}
?>