<?php
/**
 * get_my_items.php - جلب العناصر المشتراة للمستخدم
 * =====================================================
 * يجلب جميع العناصر التي اشتراها المستخدم من المتجر
 * مع معلومات عن حالة التطبيق
 * 
 * @version 1.0
 * @date 2026-02-14
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL);
ini_set('display_errors', 0);

require_once 'db_connect.php';

// استقبال البيانات
$input = json_decode(file_get_contents('php://input'), true);
$userId = $input['user_id'] ?? $_POST['user_id'] ?? 0;

// التحقق من صحة البيانات
if ($userId <= 0) {
    http_response_code(400);
    exit(json_encode([
        'success' => false,
        'status' => 'error', 
        'message' => 'معرف المستخدم مطلوب'
    ]));
}

try {
    // ========================================
    // 1. التحقق من وجود المستخدم
    // ========================================
    $stmt = $conn->prepare("SELECT id, display_name FROM users WHERE id = :uid");
    $stmt->execute([':uid' => $userId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        http_response_code(404);
        exit(json_encode([
            'success' => false,
            'status' => 'error', 
            'message' => 'المستخدم غير موجود'
        ]));
    }
    
    // ========================================
    // 2. محاولة جلب من الكاش
    // ========================================
    $cacheKey = "my_items_user_{$userId}";
    $cacheTime = 300; // 5 دقائق
    $useCache = false;
    
    if (function_exists('apcu_fetch')) {
        $cached = apcu_fetch($cacheKey, $success);
        if ($success) {
            echo json_encode($cached);
            exit;
        }
        $useCache = true;
    }
    
    // ========================================
    // 3. جلب العناصر المشتراة مع حالة التطبيق
    // ========================================
    $stmt = $conn->prepare("
        SELECT 
            si.id,
            si.item_type,
            si.item_identifier,
            si.item_name,
            si.item_name_ar,
            si.item_description,
            si.item_preview_url,
            si.rarity,
            si.category,
            
            -- تاريخ الشراء
            uui.unlocked_at,
            
            -- التحقق من التطبيق الحالي
            CASE 
                WHEN uai.id IS NOT NULL THEN TRUE 
                ELSE FALSE 
            END as is_applied,
            
            -- تاريخ التطبيق
            uai.applied_at,
            
            -- التحقق من المفضلة
            CASE 
                WHEN ufi.id IS NOT NULL THEN TRUE 
                ELSE FALSE 
            END as is_favorite
            
        FROM user_unlocked_items uui
        JOIN shop_items si ON uui.item_id = si.id
        LEFT JOIN user_applied_items uai 
            ON si.id = uai.item_id AND uai.user_id = :uid
        LEFT JOIN user_favorite_items ufi 
            ON si.id = ufi.item_id AND ufi.user_id = :uid
        WHERE uui.user_id = :uid
        AND si.is_active = TRUE
        ORDER BY 
            si.item_type,
            CASE si.rarity
                WHEN 'legendary' THEN 1
                WHEN 'epic' THEN 2
                WHEN 'rare' THEN 3
                WHEN 'common' THEN 4
                ELSE 5
            END,
            uui.unlocked_at DESC
    ");
    
    $stmt->execute([':uid' => $userId]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // ========================================
    // 4. تنظيم البيانات حسب النوع
    // ========================================
    $organized = [
        'frames' => [],
        'fonts' => [],
        'colors' => []
    ];
    
    $stats = [
        'total_items' => count($items),
        'applied_items' => 0,
        'favorite_items' => 0,
        'by_type' => [
            'frames' => 0,
            'fonts' => 0,
            'colors' => 0
        ],
        'by_rarity' => [
            'legendary' => 0,
            'epic' => 0,
            'rare' => 0,
            'common' => 0
        ]
    ];
    
    foreach ($items as $item) {
        // تحويل القيم البوليانية
        $item['is_applied'] = (bool)$item['is_applied'];
        $item['is_favorite'] = (bool)$item['is_favorite'];
        
        // تحويل الأرقام
        $item['id'] = (int)$item['id'];
        
        // إحصائيات
        if ($item['is_applied']) $stats['applied_items']++;
        if ($item['is_favorite']) $stats['favorite_items']++;
        
        $type = $item['item_type'] . 's';
        $stats['by_type'][$type]++;
        
        $rarity = $item['rarity'] ?? 'common';
        if (isset($stats['by_rarity'][$rarity])) {
            $stats['by_rarity'][$rarity]++;
        }
        
        // تنظيم حسب النوع
        $organized[$type][] = $item;
    }
    
    // ========================================
    // 5. بناء الاستجابة
    // ========================================
    $response = [
        'success' => true,
        'status' => 'success',
        'user' => [
            'id' => (int)$user['id'],
            'display_name' => $user['display_name']
        ],
        'items' => $organized,
        'stats' => $stats,
        'cache_enabled' => $useCache
    ];
    
    // ========================================
    // 6. حفظ في الكاش
    // ========================================
    if ($useCache) {
        apcu_store($cacheKey, $response, $cacheTime);
    }
    
    // ========================================
    // 7. إرسال الاستجابة
    // ========================================
    echo json_encode($response);

} catch (PDOException $e) {
    error_log("My Items Error (DB): " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'status' => 'error',
        'message' => 'خطأ في قاعدة البيانات',
        'error_code' => 'DB_ERROR'
    ]);
    
} catch (Exception $e) {
    error_log("My Items Error: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'status' => 'error',
        'message' => 'خطأ في السيرفر',
        'error_code' => 'SERVER_ERROR'
    ]);
}
?>