<?php
// get_love_meter_stats.php - الإصدار المطور (v2)
// 1. يجعل الإحصائيات شاملة لكل أنواع الإشارات.
// 2. يضيف قسماً جديداً لإحصائيات تفصيلية لكل نوع إشارة.

header('Content-Type: application/json');
ob_start();

require 'db_connect.php';

$response_data = [];

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Only POST method is allowed', 405);
    }

    $data = json_decode(file_get_contents('php://input'), true);
    $user_id = $data['user_id'] ?? null;

    if (empty($user_id)) {
        throw new Exception('User ID is required.', 400);
    }

    // 1. جلب couple_id
    $stmt_couple = $conn->prepare("SELECT couple_id FROM users WHERE id = :user_id");
    $stmt_couple->execute([':user_id' => $user_id]);
    $couple_id = $stmt_couple->fetchColumn();

    if (!$couple_id) {
        throw new Exception('User is not in a relationship.', 404);
    }

    // ===== بداية التعديل (v2): تعديل الاستعلامات لتشمل كل أنواع الإشارات =====

    // 2. جلب بيانات الرسم البياني (لكل أنواع الإشارات)
    $seven_days_ago = date('Y-m-d', strtotime('-6 days'));
    $stmt_chart = $conn->prepare(
        "SELECT 
            DAYNAME(created_at) as day_name_en, 
            COUNT(*) as interaction_count
         FROM signals 
         WHERE couple_id = :couple_id 
           AND created_at >= :start_date
         GROUP BY DAYOFWEEK(created_at), day_name_en" // تم حذف شرط `signal_type`
    );
    $stmt_chart->execute([':couple_id' => $couple_id, ':start_date' => $seven_days_ago]);
    $chart_data_raw = $stmt_chart->fetchAll(PDO::FETCH_KEY_PAIR);

    $days_map_en_to_ar = [
        'Saturday' => 'سبت', 'Sunday' => 'أحد', 'Monday' => 'اثنين',
        'Tuesday' => 'ثلاثاء', 'Wednesday' => 'أربعاء', 'Thursday' => 'خميس', 'Friday' => 'جمعة'
    ];
    
    $chart_data_final = [];
    foreach ($chart_data_raw as $day_en => $count) {
        if (isset($days_map_en_to_ar[$day_en])) {
            $day_ar = $days_map_en_to_ar[$day_en];
            $chart_data_final[$day_ar] = $count;
        }
    }

    // 3. حساب إجمالي التفاعلات في آخر 7 أيام
    $weekly_interactions = array_sum($chart_data_final);

    // 4. حساب اليوم الأكثر نشاطًا
    $busiest_day_arabic = 'غير محدد';
    if (!empty($chart_data_final)) {
        $busiest_day_arabic = array_search(max($chart_data_final), $chart_data_final);
    }

    // 5. حساب المتوسط اليومي (لكل أنواع الإشارات)
    $stmt_start_date = $conn->prepare("SELECT start_date FROM couples WHERE id = :couple_id");
    $stmt_start_date->execute([':couple_id' => $couple_id]);
    $start_date_str = $stmt_start_date->fetchColumn();
    $total_days = $start_date_str ? (new DateTime())->diff(new DateTime($start_date_str))->days + 1 : 1;

    // جلب العدد الإجمالي من جدول `couples` مباشرة (أكثر كفاءة)
    $stmt_total_interactions = $conn->prepare("SELECT love_counter FROM couples WHERE id = :couple_id");
    $stmt_total_interactions->execute([':couple_id' => $couple_id]);
    $total_interactions = (int) $stmt_total_interactions->fetchColumn();
    $daily_average = ($total_days > 0) ? round($total_interactions / $total_days, 1) : 0;

    // 6. [إضافة جديدة] جلب الإحصائيات التفصيلية لكل نوع
    $stmt_detailed_stats = $conn->prepare(
        "SELECT love_signals_count, sad_signals_count, missing_signals_count 
         FROM couples WHERE id = :couple_id"
    );
    $stmt_detailed_stats->execute([':couple_id' => $couple_id]);
    $detailed_counts = $stmt_detailed_stats->fetch(PDO::FETCH_ASSOC);

    // ===== نهاية التعديل (v2) =====

    // تجميع الرد النهائي
    $response_data = [
        'status' => 'success',
        'stats' => [
            'weekly_interactions' => $weekly_interactions, // اسم متغير أوضح
            'busiest_day' => $busiest_day_arabic,
            'daily_average' => $daily_average,
            'weekly_chart_data' => $chart_data_final,
            'detailed_counts' => [ // <-- إضافة القسم الجديد هنا
                'love' => (int) ($detailed_counts['love_signals_count'] ?? 0),
                'sad' => (int) ($detailed_counts['sad_signals_count'] ?? 0),
                'missing' => (int) ($detailed_counts['missing_signals_count'] ?? 0)
            ]
        ]
    ];

} catch (Exception $e) {
    $code = in_array($e->getCode(), [400, 404, 405]) ? $e->getCode() : 500;
    http_response_code($code );
    $response_data = ['status' => 'error', 'message' => $e->getMessage()];
}

ob_end_clean();
echo json_encode($response_data);
exit();
?>
