<?php
/**
 * get_favorite_items.php - جلب العناصر المفضلة للمستخدم
 * =====================================================
 * يجلب جميع العناصر التي أضافها المستخدم للمفضلة
 * سواء كانت مشتراة أو لم تشترى بعد
 * 
 * @version 1.0
 * @date 2026-02-14
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL);
ini_set('display_errors', 0);

require_once 'db_connect.php';

// استقبال البيانات
$input = json_decode(file_get_contents('php://input'), true);
$userId = $input['user_id'] ?? $_POST['user_id'] ?? 0;

// التحقق من صحة البيانات
if ($userId <= 0) {
    http_response_code(400);
    exit(json_encode([
        'success' => false,
        'status' => 'error', 
        'message' => 'معرف المستخدم مطلوب'
    ]));
}

try {
    // ========================================
    // 1. التحقق من وجود المستخدم وجلب بياناته
    // ========================================
    $stmt = $conn->prepare(
        "SELECT id, level, coins, display_name FROM users WHERE id = :uid"
    );
    $stmt->execute([':uid' => $userId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        http_response_code(404);
        exit(json_encode([
            'success' => false,
            'status' => 'error', 
            'message' => 'المستخدم غير موجود'
        ]));
    }
    
    $userLevel = (int)$user['level'];
    $userCoins = (int)$user['coins'];
    
    // ========================================
    // 2. محاولة جلب من الكاش
    // ========================================
    $cacheKey = "favorite_items_user_{$userId}_v2";
    $cacheTime = 300; // 5 دقائق
    $useCache = false;
    
    if (function_exists('apcu_fetch')) {
        $cached = apcu_fetch($cacheKey, $success);
        if ($success) {
            // التحقق من أن بيانات المستخدم لم تتغير
            if ($cached['user_level'] == $userLevel && 
                $cached['user_coins'] == $userCoins) {
                echo json_encode($cached);
                exit;
            }
        }
        $useCache = true;
    }
    
    // ========================================
    // 3. جلب العناصر المفضلة
    // ========================================
    $stmt = $conn->prepare("
        SELECT 
            si.id,
            si.item_type,
            si.item_identifier,
            si.item_name,
            si.item_name_ar,
            si.item_description,
            si.item_preview_url,
            si.unlock_type,
            si.coins_required,
            si.level_required,
            si.rarity,
            si.category,
            si.is_on_sale,
            si.discount_percentage,
            
            -- حساب السعر الفعلي
            CASE 
                WHEN si.is_on_sale = TRUE 
                    AND si.discount_percentage > 0 
                    AND (si.sale_end_date IS NULL OR si.sale_end_date > NOW())
                THEN FLOOR(si.coins_required * (100 - si.discount_percentage) / 100)
                ELSE si.coins_required
            END as final_price,
            
            -- التحقق من فتح العنصر
            CASE 
                WHEN uui.id IS NOT NULL THEN TRUE 
                ELSE FALSE 
            END as is_unlocked,
            
            -- التحقق من التطبيق
            CASE 
                WHEN uai.id IS NOT NULL THEN TRUE 
                ELSE FALSE 
            END as is_applied,
            
            -- التحقق من إمكانية الشراء
            CASE 
                WHEN uui.id IS NOT NULL THEN FALSE
                WHEN si.unlock_type = 'free' THEN TRUE
                WHEN si.unlock_type = 'coins' THEN
                    CASE 
                        WHEN si.is_on_sale = TRUE 
                            AND si.discount_percentage > 0 
                            AND (si.sale_end_date IS NULL OR si.sale_end_date > NOW())
                        THEN :coins >= FLOOR(si.coins_required * (100 - si.discount_percentage) / 100)
                        ELSE :coins >= si.coins_required
                    END
                WHEN si.unlock_type = 'level' THEN :level >= si.level_required
                WHEN si.unlock_type = 'both' THEN
                    CASE 
                        WHEN si.is_on_sale = TRUE 
                            AND si.discount_percentage > 0 
                            AND (si.sale_end_date IS NULL OR si.sale_end_date > NOW())
                        THEN (:coins >= FLOOR(si.coins_required * (100 - si.discount_percentage) / 100) 
                              AND :level >= si.level_required)
                        ELSE (:coins >= si.coins_required AND :level >= si.level_required)
                    END
                ELSE FALSE
            END as can_unlock,
            
            -- تاريخ الإضافة للمفضلة
            ufi.created_at as favorited_at
            
        FROM user_favorite_items ufi
        JOIN shop_items si ON ufi.item_id = si.id
        LEFT JOIN user_unlocked_items uui 
            ON si.id = uui.item_id AND uui.user_id = :uid
        LEFT JOIN user_applied_items uai 
            ON si.id = uai.item_id AND uai.user_id = :uid
        WHERE ufi.user_id = :uid
        AND si.is_active = TRUE
        ORDER BY 
            si.item_type,
            CASE si.rarity
                WHEN 'legendary' THEN 1
                WHEN 'epic' THEN 2
                WHEN 'rare' THEN 3
                WHEN 'common' THEN 4
                ELSE 5
            END,
            ufi.created_at DESC
    ");
    
    $stmt->execute([
        ':uid' => $userId,
        ':coins' => $userCoins,
        ':level' => $userLevel
    ]);
    
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // ========================================
    // 4. تنظيم البيانات حسب النوع
    // ========================================
    $organized = [
        'frames' => [],
        'fonts' => [],
        'colors' => []
    ];
    
    $stats = [
        'total_favorites' => count($items),
        'unlocked_favorites' => 0,
        'can_buy_now' => 0,
        'need_more_coins' => 0,
        'need_higher_level' => 0,
        'by_type' => [
            'frames' => 0,
            'fonts' => 0,
            'colors' => 0
        ]
    ];
    
    foreach ($items as $item) {
        // تحويل القيم البوليانية
        $item['is_unlocked'] = (bool)$item['is_unlocked'];
        $item['is_applied'] = (bool)$item['is_applied'];
        $item['can_unlock'] = (bool)$item['can_unlock'];
        $item['is_on_sale'] = (bool)$item['is_on_sale'];
        
        // تحويل الأرقام
        $item['id'] = (int)$item['id'];
        $item['coins_required'] = (int)$item['coins_required'];
        $item['level_required'] = (int)$item['level_required'];
        $item['final_price'] = (int)$item['final_price'];
        $item['discount_percentage'] = (int)$item['discount_percentage'];
        
        // إضافة حالة العنصر
        if ($item['is_unlocked']) {
            $item['status'] = 'unlocked';
            $stats['unlocked_favorites']++;
        } elseif ($item['can_unlock']) {
            $item['status'] = 'can_buy';
            $stats['can_buy_now']++;
        } else {
            if ($userCoins < $item['final_price']) {
                $item['status'] = 'insufficient_coins';
                $stats['need_more_coins']++;
            } elseif ($userLevel < $item['level_required']) {
                $item['status'] = 'insufficient_level';
                $stats['need_higher_level']++;
            } else {
                $item['status'] = 'locked';
            }
        }
        
        // المفضلة = true دائماً
        $item['is_favorite'] = true;
        
        // إحصائيات
        $type = $item['item_type'] . 's';
        $stats['by_type'][$type]++;
        
        // تنظيم حسب النوع
        $organized[$type][] = $item;
    }
    
    // ========================================
    // 5. بناء الاستجابة
    // ========================================
    $response = [
        'success' => true,
        'status' => 'success',
        'user_level' => $userLevel,
        'user_coins' => $userCoins,
        'items' => $organized,
        'stats' => $stats,
        'cache_enabled' => $useCache
    ];
    
    // ========================================
    // 6. حفظ في الكاش
    // ========================================
    if ($useCache) {
        apcu_store($cacheKey, $response, $cacheTime);
    }
    
    // ========================================
    // 7. إرسال الاستجابة
    // ========================================
    echo json_encode($response);

} catch (PDOException $e) {
    error_log("Favorite Items Error (DB): " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'status' => 'error',
        'message' => 'خطأ في قاعدة البيانات',
        'error_code' => 'DB_ERROR'
    ]);
    
} catch (Exception $e) {
    error_log("Favorite Items Error: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'status' => 'error',
        'message' => 'خطأ في السيرفر',
        'error_code' => 'SERVER_ERROR'
    ]);
}
?>