<?php
// get_dashboard_data.php
header('Content-Type: application/json; charset=utf-8');
require 'db_connect.php';

function calculateDistance($lat1, $lon1, $lat2, $lon2) {
    if ($lat1 === null || $lon1 === null || $lat2 === null || $lon2 === null) return null;
    if (($lat1 == $lat2) && ($lon1 == $lon2)) return 0;
    $theta = $lon1 - $lon2;
    $dist = sin(deg2rad($lat1)) * sin(deg2rad($lat2)) + cos(deg2rad($lat1)) * cos(deg2rad($lat2)) * cos(deg2rad($theta));
    $dist = acos($dist);
    $dist = rad2deg($dist);
    $miles = $dist * 60 * 1.1515;
    return round($miles * 1.609344, 2);
}

$data = json_decode(file_get_contents('php://input'), true);
$user_id = $data['user_id'] ?? 0;

if (empty($user_id)) {
    http_response_code(400);
    exit(json_encode(['status' => 'error', 'message' => 'User ID is required.']));
}

try {
    $stmt = $conn->prepare(
       "SELECT 
            u1.display_name,
            u1.profile_image_url AS user_image_url,
            u1.latitude AS user_lat, 
            u1.longitude AS user_lon,
            u1.name_color AS user_name_color,
            u1.font_identifier AS user_font_identifier,
            u1.frame_identifier AS user_frame_identifier,
            u1.is_verified AS user_is_verified,
            u1.level AS user_level,
            u1.xp AS user_xp,
            u1.coins AS user_coins,
            u1.vip_tier AS user_vip_tier,
            u1.vip_expires_at AS user_vip_expires_at,

            p.id AS partner_id,
            p.display_name AS partner_display_name,
            p.profile_image_url AS partner_image_url,
            p.latitude AS partner_lat,
            p.longitude AS partner_lon,
            p.name_color AS partner_name_color,
            p.font_identifier AS partner_font_identifier,
            p.frame_identifier AS partner_frame_identifier,
            p.is_verified AS partner_is_verified,
            p.vip_tier AS partner_vip_tier,
            p.vip_expires_at AS partner_vip_expires_at,

            c.id AS couple_id,
            c.start_date,
            (c.love_signals_count * 5) AS love_points,
            c.love_signals_count,
            c.sad_signals_count,
            c.missing_signals_count
        FROM users u1
        LEFT JOIN couples c ON u1.couple_id = c.id
        LEFT JOIN users p ON (c.user1_id = p.id OR c.user2_id = p.id) AND p.id != u1.id
        WHERE u1.id = :user_id"
    );
    
    $stmt->execute([':user_id' => $user_id]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    // ✅ دالة مساعدة لحساب VIP
    $calcVip = function($tier, $expiresAt) {
        $isActive = (
            $tier !== null &&
            $tier !== 'none' &&
            ($expiresAt === null || strtotime($expiresAt) > time())
        );
        return [
            'vip_tier'      => $tier ?? 'none',
            'vip_is_active' => $isActive,
        ];
    };

    $response_data = [];

    if (!$result || $result['partner_id'] === null) {
        $stmt_user = $conn->prepare(
            "SELECT display_name, profile_image_url, name_color, font_identifier,
                    frame_identifier, is_verified, level, xp, coins,
                    vip_tier, vip_expires_at
             FROM users WHERE id = :id"
        );
        $stmt_user->execute([':id' => $user_id]);
        $user_only = $stmt_user->fetch(PDO::FETCH_ASSOC);

        $userVip = $calcVip($user_only['vip_tier'] ?? null, $user_only['vip_expires_at'] ?? null);

        $response_data = [
            'display_name'          => $user_only['display_name'] ?? null,
            'user_image_url'        => $user_only['profile_image_url'] ?? null,
            'user_name_color'       => $user_only['name_color'] ?? null,
            'user_font_identifier'  => $user_only['font_identifier'] ?? null,
            'user_frame_identifier' => $user_only['frame_identifier'] ?? null,
            'user_is_verified'      => (bool)($user_only['is_verified'] ?? false),
            'user_level'            => (int)($user_only['level'] ?? 1),
            'user_xp'               => (int)($user_only['xp'] ?? 0),
            'user_coins'            => (int)($user_only['coins'] ?? 0),
            // ✅ VIP المستخدم
            'user_vip_tier'         => $userVip['vip_tier'],
            'user_vip_is_active'    => $userVip['vip_is_active'],

            'partner_id'                => null,
            'couple_id'                 => null,
            'partner_display_name'      => null,
            'partner_image_url'         => null,
            'partner_name_color'        => null,
            'partner_font_identifier'   => null,
            'partner_frame_identifier'  => null,
            'partner_is_verified'       => false,
            // ✅ VIP الشريك (فارغ)
            'partner_vip_tier'          => 'none',
            'partner_vip_is_active'     => false,

            'start_date'            => null,
            'love_points'           => 0,
            'love_signals_count'    => 0,
            'sad_signals_count'     => 0,
            'missing_signals_count' => 0,
            'distance_km'           => null,
        ];
    } else {
        $userVip    = $calcVip($result['user_vip_tier'] ?? null,    $result['user_vip_expires_at'] ?? null);
        $partnerVip = $calcVip($result['partner_vip_tier'] ?? null, $result['partner_vip_expires_at'] ?? null);

        $response_data = [
            'display_name'          => $result['display_name'],
            'user_image_url'        => $result['user_image_url'],
            'user_name_color'       => $result['user_name_color'],
            'user_font_identifier'  => $result['user_font_identifier'],
            'user_frame_identifier' => $result['user_frame_identifier'],
            'user_is_verified'      => (bool)$result['user_is_verified'],
            'user_level'            => (int)($result['user_level'] ?? 1),
            'user_xp'               => (int)($result['user_xp'] ?? 0),
            'user_coins'            => (int)($result['user_coins'] ?? 0),
            // ✅ VIP المستخدم
            'user_vip_tier'         => $userVip['vip_tier'],
            'user_vip_is_active'    => $userVip['vip_is_active'],

            'partner_id'                => (int)$result['partner_id'],
            'couple_id'                 => (int)$result['couple_id'],
            'partner_display_name'      => $result['partner_display_name'],
            'partner_image_url'         => $result['partner_image_url'],
            'partner_name_color'        => $result['partner_name_color'],
            'partner_font_identifier'   => $result['partner_font_identifier'],
            'partner_frame_identifier'  => $result['partner_frame_identifier'],
            'partner_is_verified'       => (bool)$result['partner_is_verified'],
            // ✅ VIP الشريك
            'partner_vip_tier'          => $partnerVip['vip_tier'],
            'partner_vip_is_active'     => $partnerVip['vip_is_active'],

            'start_date'            => $result['start_date'],
            'love_points'           => (int)$result['love_points'],
            'love_signals_count'    => (int)$result['love_signals_count'],
            'sad_signals_count'     => (int)$result['sad_signals_count'],
            'missing_signals_count' => (int)$result['missing_signals_count'],
            'distance_km'           => calculateDistance(
                $result['user_lat'],
                $result['user_lon'],
                $result['partner_lat'],
                $result['partner_lon']
            ),
        ];
    }

    $stmt_requests = $conn->prepare(
        "SELECT cr.id as request_id, u.display_name as requester_name 
         FROM connection_requests cr JOIN users u ON cr.requester_id = u.id
         WHERE cr.recipient_id = :current_user_id AND cr.status = 'pending'"
    );
    $stmt_requests->execute([':current_user_id' => $user_id]);
    $response_data['pending_requests'] = $stmt_requests->fetchAll(PDO::FETCH_ASSOC);

    $stmt_unread = $conn->prepare(
        "SELECT COUNT(*) as total_unread FROM messages WHERE recipient_id = :user_id AND is_read = 0"
    );
    $stmt_unread->execute([':user_id' => $user_id]);
    $unread_result = $stmt_unread->fetch(PDO::FETCH_ASSOC);
    $response_data['total_unread_count'] = (int)$unread_result['total_unread'];

    echo json_encode(['status' => 'success', 'data' => $response_data]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}
?>