<?php
/**
 * get_achievements_data.php
 * ========================
 * جلب جميع إحصائيات وإنجازات المستخدم
 * 
 * يشمل:
 * - تاريخ التسجيل (created_at)
 * - المستوى والخبرة والعملات (level, xp, coins)
 * - عدد الرسائل المرسلة
 * - عدد الإشارات (حب، حزن، اشتياق)
 * - نقاط الحب
 * - تاريخ بدء العلاقة
 * - الوقت الإجمالي في التطبيق
 * - عدد زيارات البروفايل
 * 
 * @version 1.0
 * @date 2026-02-13
 */

header('Content-Type: application/json; charset=utf-8');
require 'db_connect.php';

// قراءة البيانات من JSON body
$data = json_decode(file_get_contents('php://input'), true);
$user_id = isset($data['user_id']) ? intval($data['user_id']) : 0;

// التحقق من صحة البيانات
if (empty($user_id)) {
    http_response_code(400);
    exit(json_encode([
        'status' => 'error', 
        'message' => 'User ID is required.'
    ]));
}

try {
    // ═══════════════════════════════════════════════
    //          1. البيانات الأساسية للمستخدم
    // ═══════════════════════════════════════════════
    $stmt_user = $conn->prepare(
        "SELECT 
            id,
            created_at,
            level,
            xp,
            coins,
            couple_id
        FROM users 
        WHERE id = :user_id"
    );
    
    $stmt_user->execute([':user_id' => $user_id]);
    $user = $stmt_user->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        throw new Exception("User not found.");
    }

    // ═══════════════════════════════════════════════
    //          2. عدد الرسائل المرسلة
    // ═══════════════════════════════════════════════
    $stmt_messages = $conn->prepare(
        "SELECT COUNT(*) as messages_sent 
         FROM messages 
         WHERE sender_id = :user_id"
    );
    $stmt_messages->execute([':user_id' => $user_id]);
    $messages_data = $stmt_messages->fetch(PDO::FETCH_ASSOC);
    $messages_sent = (int)$messages_data['messages_sent'];

    // ═══════════════════════════════════════════════
    //          3. بيانات العلاقة والإشارات
    // ═══════════════════════════════════════════════
    $couple_data = null;
    $partner_display_name = null;
    
    if ($user['couple_id']) {
        $stmt_couple = $conn->prepare(
            "SELECT 
                c.start_date,
                c.love_signals_count,
                c.sad_signals_count,
                c.missing_signals_count,
                p.display_name as partner_display_name
            FROM couples c
            LEFT JOIN users p ON (
                (c.user1_id = p.id OR c.user2_id = p.id) 
                AND p.id != :user_id
            )
            WHERE c.id = :couple_id"
        );
        
        $stmt_couple->execute([
            ':couple_id' => $user['couple_id'],
            ':user_id' => $user_id
        ]);
        
        $couple_data = $stmt_couple->fetch(PDO::FETCH_ASSOC);
        
        if ($couple_data) {
            $partner_display_name = $couple_data['partner_display_name'];
        }
    }

    // ═══════════════════════════════════════════════
    //          4. نقاط الحب (love_signals * 5)
    // ═══════════════════════════════════════════════
    $love_points = 0;
    if ($couple_data) {
        $love_points = (int)$couple_data['love_signals_count'] * 5;
    }

    // ═══════════════════════════════════════════════
    //          5. الوقت الإجمالي في التطبيق
    // ═══════════════════════════════════════════════
    // حساب الوقت من جدول daily_actions_log حيث action_type = 'spend_time_in_app'
    // كل سجل يمثل دقيقة واحدة (60 ثانية)
    $stmt_usage = $conn->prepare(
        "SELECT COUNT(*) as total_minutes
         FROM daily_actions_log 
         WHERE user_id = :user_id 
         AND action_type = 'spend_time_in_app'"
    );
    $stmt_usage->execute([':user_id' => $user_id]);
    $usage_data = $stmt_usage->fetch(PDO::FETCH_ASSOC);
    $total_minutes = (int)$usage_data['total_minutes'];
    $total_usage_seconds = $total_minutes * 60;

    // ═══════════════════════════════════════════════
    //          6. عدد زيارات البروفايل
    // ═══════════════════════════════════════════════
    $stmt_visits = $conn->prepare(
        "SELECT COUNT(DISTINCT visitor_id) as profile_visits
         FROM profile_visits 
         WHERE visited_id = :user_id"
    );
    $stmt_visits->execute([':user_id' => $user_id]);
    $visits_data = $stmt_visits->fetch(PDO::FETCH_ASSOC);
    $profile_visits = (int)$visits_data['profile_visits'];

    // ═══════════════════════════════════════════════
    //          7. تجميع البيانات النهائية
    // ═══════════════════════════════════════════════
    $response_data = [
        // معلومات العضوية
        'created_at' => $user['created_at'],
        'level' => (int)$user['level'],
        'xp' => (int)$user['xp'],
        'coins' => (int)$user['coins'],
        
        // الرسائل
        'messages_sent' => $messages_sent,
        
        // الإشارات
        'love_signals_count' => $couple_data ? (int)$couple_data['love_signals_count'] : 0,
        'sad_signals_count' => $couple_data ? (int)$couple_data['sad_signals_count'] : 0,
        'missing_signals_count' => $couple_data ? (int)$couple_data['missing_signals_count'] : 0,
        
        // نقاط الحب
        'love_points' => $love_points,
        
        // بيانات العلاقة
        'start_date' => $couple_data ? $couple_data['start_date'] : null,
        'partner_display_name' => $partner_display_name,
        
        // النشاط
        'total_usage_seconds' => $total_usage_seconds,
        'profile_visits' => $profile_visits
    ];

    echo json_encode([
        'status' => 'success',
        'data' => $response_data
    ]);

} catch (Exception $e) {
    http_response_code(500);
    error_log('Get Achievements Error: ' . $e->getMessage());
    echo json_encode([
        'status' => 'error', 
        'message' => 'خطأ في الخادم: ' . $e->getMessage()
    ]);
}
?>