<?php
/**
 * daily_stats_report.php
 * ======================
 * تقرير إحصائيات يومي عن نظام المكافآت
 * 
 * يجب تشغيله يومياً عبر Cron Job
 * 
 * @version 1.0
 * @date 2026-02-05
 */

require 'db_connect.php';

try {
    $today = date('Y-m-d');
    $yesterday = date('Y-m-d', strtotime('-1 day'));
    
    echo "╔═══════════════════════════════════════════════════════╗\n";
    echo "║       تقرير إحصائيات المكافآت اليومية                 ║\n";
    echo "║       التاريخ: $today                       ║\n";
    echo "╚═══════════════════════════════════════════════════════╝\n\n";
    
    // 1. إحصائيات اليوم
    echo "📊 إحصائيات اليوم ($today):\n";
    echo "═══════════════════════════════════════════════════════\n";
    
    $stmt = $conn->prepare(
        "SELECT 
            COUNT(DISTINCT user_id) as active_users,
            COUNT(*) as total_actions,
            SUM(coins_earned) as total_coins,
            SUM(xp_earned) as total_xp
         FROM daily_actions_log 
         WHERE DATE(action_date) = :today"
    );
    $stmt->execute([':today' => $today]);
    $today_stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    echo "  المستخدمين النشطين: " . $today_stats['active_users'] . "\n";
    echo "  إجمالي الأنشطة: " . $today_stats['total_actions'] . "\n";
    echo "  العملات الممنوحة: " . $today_stats['total_coins'] . " 🪙\n";
    echo "  نقاط الخبرة الممنوحة: " . $today_stats['total_xp'] . " ⭐\n\n";
    
    // 2. توزيع الأنشطة
    echo "📈 توزيع الأنشطة اليوم:\n";
    echo "═══════════════════════════════════════════════════════\n";
    
    $stmt = $conn->prepare(
        "SELECT 
            action_type,
            COUNT(*) as count,
            COUNT(DISTINCT user_id) as unique_users,
            SUM(coins_earned) as coins,
            SUM(xp_earned) as xp
         FROM daily_actions_log 
         WHERE DATE(action_date) = :today
         GROUP BY action_type
         ORDER BY count DESC"
    );
    $stmt->execute([':today' => $today]);
    $activities = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $action_names = [
        'daily_login' => 'تسجيل الدخول',
        'send_signal' => 'إرسال إشارة',
        'add_memory' => 'إضافة ذكرى',
        'send_message' => 'إرسال رسالة',
        'truth_game_round' => 'جولة لعبة الصراحة',
        'spend_time_in_app' => 'وقت في التطبيق'
    ];
    
    foreach ($activities as $activity) {
        $name = $action_names[$activity['action_type']] ?? $activity['action_type'];
        echo "  • $name:\n";
        echo "     - العدد: " . $activity['count'] . "\n";
        echo "     - المستخدمين: " . $activity['unique_users'] . "\n";
        echo "     - العملات: " . $activity['coins'] . " 🪙\n";
        echo "     - الخبرة: " . $activity['xp'] . " ⭐\n";
    }
    echo "\n";
    
    // 3. أكثر المستخدمين نشاطاً
    echo "🏆 أكثر 10 مستخدمين نشاطاً اليوم:\n";
    echo "═══════════════════════════════════════════════════════\n";
    
    $stmt = $conn->prepare(
        "SELECT 
            u.id,
            u.username,
            u.display_name,
            COUNT(*) as actions,
            SUM(dal.coins_earned) as coins,
            SUM(dal.xp_earned) as xp
         FROM daily_actions_log dal
         JOIN users u ON dal.user_id = u.id
         WHERE DATE(dal.action_date) = :today
         GROUP BY u.id
         ORDER BY actions DESC
         LIMIT 10"
    );
    $stmt->execute([':today' => $today]);
    $top_users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $rank = 1;
    foreach ($top_users as $user) {
        echo "  $rank. " . $user['display_name'] . " (@" . $user['username'] . ")\n";
        echo "     - الأنشطة: " . $user['actions'] . "\n";
        echo "     - العملات: " . $user['coins'] . " 🪙\n";
        echo "     - الخبرة: " . $user['xp'] . " ⭐\n";
        $rank++;
    }
    echo "\n";
    
    // 4. المستخدمين الذين وصلوا للحدود
    echo "⚠️  المستخدمين الذين وصلوا لحدود يومية:\n";
    echo "═══════════════════════════════════════════════════════\n";
    
    $limits = [
        'daily_login' => 1,
        'send_signal' => 5,
        'add_memory' => 1,
        'truth_game_round' => 10,
        'send_message' => 10
    ];
    
    foreach ($limits as $action => $limit) {
        $stmt = $conn->prepare(
            "SELECT user_id, COUNT(*) as count
             FROM daily_actions_log
             WHERE DATE(action_date) = :today
             AND action_type = :action
             GROUP BY user_id
             HAVING count >= :limit"
        );
        $stmt->execute([
            ':today' => $today,
            ':action' => $action,
            ':limit' => $limit
        ]);
        $users_at_limit = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (count($users_at_limit) > 0) {
            $name = $action_names[$action] ?? $action;
            echo "  • $name: " . count($users_at_limit) . " مستخدم\n";
        }
    }
    echo "\n";
    
    // 5. مقارنة مع الأمس
    echo "📉 مقارنة مع الأمس ($yesterday):\n";
    echo "═══════════════════════════════════════════════════════\n";
    
    $stmt = $conn->prepare(
        "SELECT 
            COUNT(DISTINCT user_id) as active_users,
            COUNT(*) as total_actions,
            SUM(coins_earned) as total_coins,
            SUM(xp_earned) as total_xp
         FROM daily_actions_log 
         WHERE DATE(action_date) = :yesterday"
    );
    $stmt->execute([':yesterday' => $yesterday]);
    $yesterday_stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    $users_diff = $today_stats['active_users'] - $yesterday_stats['active_users'];
    $actions_diff = $today_stats['total_actions'] - $yesterday_stats['total_actions'];
    $coins_diff = $today_stats['total_coins'] - $yesterday_stats['total_coins'];
    $xp_diff = $today_stats['total_xp'] - $yesterday_stats['total_xp'];
    
    $users_change = $yesterday_stats['active_users'] > 0 
        ? round(($users_diff / $yesterday_stats['active_users']) * 100, 1) 
        : 0;
    $actions_change = $yesterday_stats['total_actions'] > 0 
        ? round(($actions_diff / $yesterday_stats['total_actions']) * 100, 1) 
        : 0;
    
    echo "  المستخدمين النشطين: ";
    echo ($users_diff >= 0 ? "+" : "") . "$users_diff ($users_change%)\n";
    
    echo "  إجمالي الأنشطة: ";
    echo ($actions_diff >= 0 ? "+" : "") . "$actions_diff ($actions_change%)\n";
    
    echo "  العملات: ";
    echo ($coins_diff >= 0 ? "+" : "") . "$coins_diff 🪙\n";
    
    echo "  الخبرة: ";
    echo ($xp_diff >= 0 ? "+" : "") . "$xp_diff ⭐\n\n";
    
    // 6. معلومات النظام
    echo "🔧 معلومات النظام:\n";
    echo "═══════════════════════════════════════════════════════\n";
    
    $stmt = $conn->query("SELECT COUNT(*) as total FROM daily_actions_log");
    $total_records = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    echo "  إجمالي السجلات في الجدول: " . number_format($total_records) . "\n";
    
    $stmt = $conn->query(
        "SELECT COUNT(*) as total 
         FROM daily_actions_log 
         WHERE action_date < DATE_SUB(NOW(), INTERVAL 30 DAY)"
    );
    $old_records = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    echo "  السجلات القديمة (+30 يوم): $old_records\n";
    
    if ($old_records > 0) {
        echo "  ⚠️  يُنصح بتشغيل cleanup_old_logs.php\n";
    }
    
    echo "\n╔═══════════════════════════════════════════════════════╗\n";
    echo "║              نهاية التقرير اليومي                     ║\n";
    echo "╚═══════════════════════════════════════════════════════╝\n";
    
    // حفظ التقرير في ملف
    $report_dir = 'reports';
    if (!is_dir($report_dir)) {
        mkdir($report_dir, 0777, true);
    }
    
    ob_start();
    include(__FILE__);
    $report_content = ob_get_clean();
    
    file_put_contents(
        "$report_dir/daily_report_$today.txt",
        $report_content
    );
    
    echo "\n✓ تم حفظ التقرير في: $report_dir/daily_report_$today.txt\n";
    
} catch (Exception $e) {
    echo "✗ خطأ: " . $e->getMessage() . "\n";
    error_log("[Daily Report Error] " . $e->getMessage());
    exit(1);
}
?>
