<?php
// ban_device.php
// ================================================
// إدارة حظر الأجهزة (Device Ban System)
// ================================================

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

try {
    $conn = new mysqli("localhost", "u323701002_us_coc", "AS_05344465928pr", "u323701002_coc");
    
    if ($conn->connect_error) {
        throw new Exception('Database connection failed: ' . $conn->connect_error);
    }
    
    $conn->set_charset("utf8mb4");

    // إنشاء جدول الحظر إذا لم يكن موجوداً
    $conn->query("
        CREATE TABLE IF NOT EXISTS banned_devices (
            id INT AUTO_INCREMENT PRIMARY KEY,
            device_id VARCHAR(255) NOT NULL UNIQUE,
            user_id INT DEFAULT NULL,
            reason TEXT DEFAULT NULL,
            banned_by VARCHAR(100) DEFAULT 'admin',
            ban_date DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            ban_expires DATETIME DEFAULT NULL,
            is_permanent TINYINT(1) DEFAULT 1,
            is_active TINYINT(1) DEFAULT 1,
            INDEX idx_device_id (device_id),
            INDEX idx_user_id (user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");

    $raw_input = file_get_contents('php://input');
    $input = json_decode($raw_input, true);

    if (!$input) {
        throw new Exception('No input data');
    }

    $action = isset($input['action']) ? $input['action'] : null;

    // -----------------------------------------------
    // 1. حظر جهاز
    // -----------------------------------------------
    if ($action === 'ban') {
        $device_id = isset($input['device_id']) ? trim($input['device_id']) : null;
        $user_id   = isset($input['user_id'])   ? intval($input['user_id']) : null;
        $reason    = isset($input['reason'])    ? trim($input['reason'])    : 'مخالفة القواعد';
        $banned_by = isset($input['banned_by']) ? trim($input['banned_by']) : 'admin';
        $is_permanent = isset($input['is_permanent']) ? intval($input['is_permanent']) : 1;
        $ban_expires  = ($is_permanent == 0 && isset($input['ban_expires']))
                        ? $input['ban_expires'] : null;

        if (!$device_id) {
            throw new Exception('device_id is required');
        }

        $stmt = $conn->prepare("
            INSERT INTO banned_devices (device_id, user_id, reason, banned_by, ban_date, ban_expires, is_permanent, is_active)
            VALUES (?, ?, ?, ?, NOW(), ?, ?, 1)
            ON DUPLICATE KEY UPDATE
                user_id      = VALUES(user_id),
                reason       = VALUES(reason),
                banned_by    = VALUES(banned_by),
                ban_date     = NOW(),
                ban_expires  = VALUES(ban_expires),
                is_permanent = VALUES(is_permanent),
                is_active    = 1
        ");
        $stmt->bind_param("sissa i", $device_id, $user_id, $reason, $banned_by, $ban_expires, $is_permanent);
        $stmt->execute();
        $stmt->close();

        echo json_encode([
            'status'  => 'success',
            'message' => 'تم حظر الجهاز بنجاح',
            'data'    => [
                'device_id'    => $device_id,
                'user_id'      => $user_id,
                'reason'       => $reason,
                'is_permanent' => $is_permanent,
                'ban_expires'  => $ban_expires
            ]
        ]);
    }

    // -----------------------------------------------
    // 2. رفع الحظر عن جهاز
    // -----------------------------------------------
    elseif ($action === 'unban') {
        $device_id = isset($input['device_id']) ? trim($input['device_id']) : null;

        if (!$device_id) {
            throw new Exception('device_id is required');
        }

        $stmt = $conn->prepare("UPDATE banned_devices SET is_active = 0 WHERE device_id = ?");
        $stmt->bind_param("s", $device_id);
        $stmt->execute();
        $stmt->close();

        echo json_encode([
            'status'  => 'success',
            'message' => 'تم رفع الحظر عن الجهاز'
        ]);
    }

    // -----------------------------------------------
    // 3. التحقق من حظر جهاز (يُستخدم من التطبيق)
    // -----------------------------------------------
    elseif ($action === 'check') {
        $device_id = isset($input['device_id']) ? trim($input['device_id']) : null;

        if (!$device_id) {
            throw new Exception('device_id is required');
        }

        $stmt = $conn->prepare("
            SELECT id, reason, ban_date, ban_expires, is_permanent
            FROM banned_devices
            WHERE device_id = ?
              AND is_active = 1
              AND (is_permanent = 1 OR ban_expires IS NULL OR ban_expires > NOW())
            LIMIT 1
        ");
        $stmt->bind_param("s", $device_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $ban    = $result->fetch_assoc();
        $stmt->close();

        if ($ban) {
            echo json_encode([
                'status'     => 'banned',
                'is_banned'  => true,
                'reason'     => $ban['reason'],
                'ban_date'   => $ban['ban_date'],
                'expires'    => $ban['ban_expires'],
                'permanent'  => (bool)$ban['is_permanent'],
                'message'    => 'تم حظر هذا الجهاز من استخدام التطبيق'
            ]);
        } else {
            echo json_encode([
                'status'    => 'ok',
                'is_banned' => false,
                'message'   => 'الجهاز غير محظور'
            ]);
        }
    }

    // -----------------------------------------------
    // 4. قائمة الأجهزة المحظورة (للوحة الإدارة)
    // -----------------------------------------------
    elseif ($action === 'list') {
        $page  = isset($input['page'])  ? max(1, intval($input['page']))  : 1;
        $limit = isset($input['limit']) ? min(100, intval($input['limit'])) : 20;
        $offset = ($page - 1) * $limit;

        // العدد الكلي
        $count_result = $conn->query("SELECT COUNT(*) as total FROM banned_devices WHERE is_active = 1");
        $total = $count_result->fetch_assoc()['total'];

        $stmt = $conn->prepare("
            SELECT bd.*, u.display_name, u.username
            FROM banned_devices bd
            LEFT JOIN users u ON bd.user_id = u.id
            WHERE bd.is_active = 1
            ORDER BY bd.ban_date DESC
            LIMIT ? OFFSET ?
        ");
        $stmt->bind_param("ii", $limit, $offset);
        $stmt->execute();
        $result = $stmt->get_result();

        $bans = [];
        while ($row = $result->fetch_assoc()) {
            $bans[] = $row;
        }
        $stmt->close();

        echo json_encode([
            'status' => 'success',
            'data'   => [
                'bans'  => $bans,
                'total' => $total,
                'page'  => $page,
                'limit' => $limit,
                'pages' => ceil($total / $limit)
            ]
        ]);
    }

    else {
        throw new Exception('Invalid action. Use: ban, unban, check, list');
    }

    $conn->close();

} catch (Exception $e) {
    error_log("Ban Device Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}
?>