<?php
/**
 * apply_shop_item.php - النسخة المحسّنة v2.1
 * ======================================
 * فتح/شراء عنصر من المتجر مع التحقق الكامل
 * + تحديث جدول users بشكل آمن
 * 
 * @version 2.1
 * @date 2026-02-15
 */

header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL);
ini_set('display_errors', 0);

require_once 'db_connect.php';

// استقبال البيانات
$input = json_decode(file_get_contents('php://input'), true);
$userId = $input['user_id'] ?? $_POST['user_id'] ?? 0;
$itemId = $input['item_id'] ?? $_POST['item_id'] ?? 0;
$action = $input['action'] ?? $_POST['action'] ?? 'apply';

// التحقق من صحة البيانات
if ($userId <= 0 || $itemId <= 0) {
    http_response_code(400);
    exit(json_encode([
        'status' => 'error', 
        'message' => 'بيانات غير صالحة. يجب توفير user_id و item_id.'
    ]));
}

if (!in_array($action, ['apply', 'unapply'])) {
    http_response_code(400);
    exit(json_encode([
        'status' => 'error', 
        'message' => 'الإجراء غير صحيح. استخدم apply أو unapply'
    ]));
}

try {
    // ========================================
    // 1. التحقق من وجود المستخدم
    // ========================================
    $stmt = $conn->prepare("SELECT id FROM users WHERE id = :uid");
    $stmt->execute([':uid' => $userId]);
    if (!$stmt->fetch()) {
        throw new Exception('المستخدم غير موجود');
    }
    
    // ========================================
    // 2. جلب بيانات العنصر
    // ========================================
    $stmt = $conn->prepare(
        "SELECT id, item_type, item_identifier, item_name_ar 
         FROM shop_items 
         WHERE id = :iid AND is_active = TRUE"
    );
    $stmt->execute([':iid' => $itemId]);
    $item = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$item) {
        throw new Exception('العنصر غير موجود أو غير متاح');
    }
    
    $itemType = $item['item_type'];
    $itemIdentifier = $item['item_identifier'];
    
    // ========================================
    // 3. التحقق من أن المستخدم اشترى العنصر
    // ========================================
    $stmt = $conn->prepare(
        "SELECT id FROM user_unlocked_items 
         WHERE user_id = :uid AND item_id = :iid"
    );
    $stmt->execute([':uid' => $userId, ':iid' => $itemId]);
    
    if (!$stmt->fetch()) {
        throw new Exception('يجب شراء العنصر أولاً! 🔒');
    }
    
    // ========================================
    // 4. تنفيذ الإجراء (تطبيق أو إلغاء)
    // ========================================
    $conn->beginTransaction();
    
    if ($action === 'apply') {
        // ----------------------------------------
        // تطبيق العنصر
        // ----------------------------------------
        
        // أولاً: حذف أي عنصر سابق من نفس النوع
        $stmt = $conn->prepare(
            "DELETE FROM user_applied_items 
             WHERE user_id = :uid AND item_type = :type"
        );
        $stmt->execute([':uid' => $userId, ':type' => $itemType]);
        
        // ثانياً: إضافة العنصر الجديد
        $stmt = $conn->prepare(
            "INSERT INTO user_applied_items (user_id, item_type, item_id) 
             VALUES (:uid, :type, :iid)"
        );
        $stmt->execute([
            ':uid' => $userId,
            ':type' => $itemType,
            ':iid' => $itemId
        ]);
        
        // ثالثاً: تحديث جدول users (بشكل آمن)
        $columnMap = [
            'frame' => 'frame_identifier',
            'font' => 'font_identifier',
            'color' => 'name_color'
        ];
        
        if (isset($columnMap[$itemType])) {
            $column = $columnMap[$itemType];
            
            // ★★★ التحقق من وجود العمود أولاً ★★★
            $stmt = $conn->prepare("
                SELECT COUNT(*) as col_exists
                FROM INFORMATION_SCHEMA.COLUMNS 
                WHERE TABLE_SCHEMA = DATABASE() 
                AND TABLE_NAME = 'users' 
                AND COLUMN_NAME = :col
            ");
            $stmt->execute([':col' => $column]);
            $colCheck = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($colCheck && $colCheck['col_exists'] > 0) {
                // العمود موجود - نفذ التحديث
                $updateSql = "UPDATE users SET `$column` = :identifier WHERE id = :uid";
                $stmt = $conn->prepare($updateSql);
                $stmt->execute([
                    ':identifier' => $itemIdentifier,
                    ':uid' => $userId
                ]);
            } else {
                // العمود غير موجود - تحذير في الـ log فقط
                error_log("Warning: Column '$column' does not exist in users table. Skipping update.");
            }
        }
        
        $message = 'تم تطبيق العنصر بنجاح! ✨';
        
    } else {
        // ----------------------------------------
        // إلغاء التطبيق
        // ----------------------------------------
        
        // أولاً: حذف من جدول user_applied_items
        $stmt = $conn->prepare(
            "DELETE FROM user_applied_items 
             WHERE user_id = :uid AND item_id = :iid"
        );
        $stmt->execute([':uid' => $userId, ':iid' => $itemId]);
        
        if ($stmt->rowCount() === 0) {
            throw new Exception('العنصر غير مطبق حالياً');
        }
        
        // ثانياً: حذف من جدول users (بشكل آمن)
        $columnMap = [
            'frame' => 'frame_identifier',
            'font' => 'font_identifier',
            'color' => 'name_color'
        ];
        
        if (isset($columnMap[$itemType])) {
            $column = $columnMap[$itemType];
            
            // ★★★ التحقق من وجود العمود أولاً ★★★
            $stmt = $conn->prepare("
                SELECT COUNT(*) as col_exists
                FROM INFORMATION_SCHEMA.COLUMNS 
                WHERE TABLE_SCHEMA = DATABASE() 
                AND TABLE_NAME = 'users' 
                AND COLUMN_NAME = :col
            ");
            $stmt->execute([':col' => $column]);
            $colCheck = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($colCheck && $colCheck['col_exists'] > 0) {
                // العمود موجود - نفذ الحذف
                $updateSql = "UPDATE users SET `$column` = NULL WHERE id = :uid";
                $stmt = $conn->prepare($updateSql);
                $stmt->execute([':uid' => $userId]);
            }
        }
        
        $message = 'تم إلغاء تطبيق العنصر';
    }
    
    // ========================================
    // 5. مسح الكاش
    // ========================================
    if (function_exists('apcu_delete')) {
        apcu_delete("shop_items_user_{$userId}_v2");
        apcu_delete("shop_items_user_{$userId}_v3");
        apcu_delete("my_items_user_{$userId}");
        apcu_delete("dashboard_data_{$userId}");
    }
    
    $conn->commit();
    
    // ========================================
    // 6. إرسال الاستجابة
    // ========================================
    echo json_encode([
        'status' => 'success',
        'message' => $message,
        'data' => [
            'action' => $action,
            'item' => [
                'id' => (int)$item['id'],
                'type' => $itemType,
                'name_ar' => $item['item_name_ar']
            ],
            'is_applied' => $action === 'apply'
        ]
    ]);

} catch (Exception $e) {
    // Rollback في حالة الخطأ
    if ($conn->inTransaction()) {
        $conn->rollBack();
    }
    
    error_log("Apply Item Error: " . $e->getMessage() . " | User: $userId | Item: $itemId");
    
    http_response_code(400);
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage(),
        'error_code' => 'APPLY_FAILED'
    ]);
}
?>