<?php
/**
 * admin_badges_api.php
 * API لإدارة الشارات من لوحة التحكم
 * ====================================
 * Actions:
 *   GET  ?action=get_badges
 *   GET  ?action=search_users&q=...
 *   GET  ?action=get_user_badges&user_id=...
 *   POST action=save_badge
 *   POST action=delete_badge
 *   POST action=assign_badges
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') exit(0);

require 'db_connect.php';

// ── قراءة الـ action ──────────────────────────────────────
$action = '';
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $action = $_GET['action'] ?? '';
} else {
    $body   = json_decode(file_get_contents('php://input'), true) ?? [];
    $action = $body['action'] ?? '';
}

// ── Router ────────────────────────────────────────────────
try {
    switch ($action) {

        // ════════════════════════════════
        //  جلب كل الشارات النشطة
        // ════════════════════════════════
        case 'get_badges':
            $st = $conn->query("
                SELECT id, name, description, icon_url, color,
                       animation_type, animation_config, sort_order
                FROM badges
                WHERE is_active = 1
                ORDER BY sort_order ASC, id ASC
            ");
            $badges = $st->fetchAll(PDO::FETCH_ASSOC);

            // animation_config: نحوله من JSON string إلى object
            foreach ($badges as &$b) {
                if (!empty($b['animation_config'])) {
                    $b['animation_config'] = json_decode($b['animation_config'], true);
                }
            }
            unset($b);

            echo json_encode(['status' => 'success', 'data' => $badges]);
            break;

        // ════════════════════════════════
        //  حفظ شارة (إنشاء أو تعديل)
        // ════════════════════════════════
        case 'save_badge':
            $id            = !empty($body['id']) ? (int)$body['id'] : 0;
            $name          = trim($body['name']          ?? '');
            $icon_url      = trim($body['icon_url']      ?? '');
            $animation_type= trim($body['animation_type']?? 'gold_stars');
            $text_color    = trim($body['text_color']    ?? '#FFD700');
            $sort_order    = (int)($body['sort_order']   ?? 0);

            if (empty($name)) {
                echo json_encode(['status' => 'error', 'message' => 'اسم الشارة مطلوب']);
                break;
            }

            // animation_config: نستخرجه من animation_type لو كان custom
            $animation_config = null;
            if (str_starts_with($animation_type, 'custom:')) {
                $jsonPart = substr($animation_type, 7);
                // نتحقق إنه JSON صحيح
                $decoded = json_decode($jsonPart, true);
                if ($decoded !== null) {
                    $animation_config = json_encode($decoded);
                    $animation_type   = 'custom';
                }
            }

            if ($id > 0) {
                // تعديل شارة موجودة
                $st = $conn->prepare("
                    UPDATE badges SET
                        name             = :name,
                        icon_url         = :icon_url,
                        color            = :color,
                        animation_type   = :animation_type,
                        animation_config = :animation_config,
                        sort_order       = :sort_order
                    WHERE id = :id
                ");
                $st->execute([
                    ':name'             => $name,
                    ':icon_url'         => $icon_url,
                    ':color'            => $text_color,
                    ':animation_type'   => $animation_type,
                    ':animation_config' => $animation_config,
                    ':sort_order'       => $sort_order,
                    ':id'               => $id,
                ]);
                echo json_encode(['status' => 'success', 'message' => '✅ تم تحديث الشارة']);
            } else {
                // إنشاء شارة جديدة
                $st = $conn->prepare("
                    INSERT INTO badges
                        (name, icon_url, color, animation_type, animation_config, sort_order, is_active)
                    VALUES
                        (:name, :icon_url, :color, :animation_type, :animation_config, :sort_order, 1)
                ");
                $st->execute([
                    ':name'             => $name,
                    ':icon_url'         => $icon_url,
                    ':color'            => $text_color,
                    ':animation_type'   => $animation_type,
                    ':animation_config' => $animation_config,
                    ':sort_order'       => $sort_order,
                ]);
                $newId = $conn->lastInsertId();
                echo json_encode(['status' => 'success', 'message' => '✅ تم إنشاء الشارة', 'id' => $newId]);
            }
            break;

        // ════════════════════════════════
        //  حذف شارة (soft delete)
        // ════════════════════════════════
        case 'delete_badge':
            $id = (int)($body['id'] ?? 0);
            if (!$id) { echo json_encode(['status' => 'error', 'message' => 'ID مطلوب']); break; }

            $conn->prepare("UPDATE badges SET is_active = 0 WHERE id = :id")
                 ->execute([':id' => $id]);

            echo json_encode(['status' => 'success', 'message' => '🗑️ تم حذف الشارة']);
            break;

        // ════════════════════════════════
        //  البحث عن مستخدمين
        // ════════════════════════════════
        case 'search_users':
            $q = trim($_GET['q'] ?? '');
            if (strlen($q) < 2) {
                echo json_encode(['status' => 'success', 'data' => []]);
                break;
            }
            $st = $conn->prepare("
                SELECT id, display_name, username, profile_image_url
                FROM users
                WHERE display_name LIKE :q OR username LIKE :q
                ORDER BY display_name ASC
                LIMIT 10
            ");
            $st->execute([':q' => '%' . $q . '%']);
            echo json_encode(['status' => 'success', 'data' => $st->fetchAll(PDO::FETCH_ASSOC)]);
            break;

        // ════════════════════════════════
        //  شارات مستخدم معين
        // ════════════════════════════════
        case 'get_user_badges':
            $uid = (int)($_GET['user_id'] ?? 0);
            if (!$uid) { echo json_encode(['status' => 'success', 'data' => []]); break; }

            $st = $conn->prepare("SELECT badge_id FROM user_badges WHERE user_id = :uid");
            $st->execute([':uid' => $uid]);
            echo json_encode(['status' => 'success', 'data' => $st->fetchAll(PDO::FETCH_ASSOC)]);
            break;

        // ════════════════════════════════
        //  تعيين شارات لمستخدم
        // ════════════════════════════════
        case 'assign_badges':
            $uid       = (int)($body['user_id']   ?? 0);
            $badgeIds  = $body['badge_ids'] ?? [];

            if (!$uid) { echo json_encode(['status' => 'error', 'message' => 'user_id مطلوب']); break; }

            // حذف القديم
            $conn->prepare("DELETE FROM user_badges WHERE user_id = :uid")
                 ->execute([':uid' => $uid]);

            // إدخال الجديد
            if (!empty($badgeIds)) {
                $st = $conn->prepare("
                    INSERT IGNORE INTO user_badges (user_id, badge_id) VALUES (:uid, :bid)
                ");
                foreach ($badgeIds as $bid) {
                    $bid = (int)$bid;
                    if ($bid > 0) $st->execute([':uid' => $uid, ':bid' => $bid]);
                }
            }

            echo json_encode(['status' => 'success', 'message' => '✅ تم حفظ الشارات']);
            break;

        // ════════════════════════════════
        //  Action غير معروف
        // ════════════════════════════════
        default:
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'action غير صحيح: ' . $action]);
            break;
    }

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'خطأ قاعدة البيانات: ' . $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}
?>